package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// RejectInvitation wraps the corresponding backend function with an authorizer
func (a *auth) RejectInvitation(ctx context.Context, invitationID, callerID string, reasonRejected models.InvitationReasonRejected) (*models.Invitation, error) {
	if !a.authorizer.CanRejectInvitation(ctx, invitationID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			InviteID:  invitationID,
			Method:    models.ErrorMethodTypeRejectInvitation,
			ErrorCode: meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot reject the invitation").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.RejectInvitation(ctx, invitationID, callerID, reasonRejected)
}

// CanRejectInvitation determines whether the caller is authorized to reject an invitation.
func (a *auth) CanRejectInvitation(ctx context.Context, invitationID, callerID string) bool {
	if callerID == "" || invitationID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	dbInvitation, err := a.backend.GetInvitationByID(ctx, invitationID)
	if err != nil || dbInvitation == nil {
		return false
	}

	if a.utils.IsInvitationRecipient(callerID, dbInvitation) {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchEditor(ctx, callerID, dbInvitation.RecipientID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
