package auth_test

import (
	"context"
	"testing"

	"github.com/pkg/errors"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanRejectInvitation(t *testing.T) {
	Convey("CanRejectInvitation", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)
		backend := (i.Backend).(*mocks.Backender)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		invitationID := testutil.MustNewID()
		recipientID := util.NewUserID()
		invitation := &models.DBInvitation{
			ID:          invitationID,
			RecipientID: recipientID,
		}
		backend.On("GetInvitationByID", mock.Anything, invitationID).Return(invitation, nil)

		Convey("No callerID should return false", func() {
			So(auth.CanRejectInvitation(ctx, "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No invitationID should return false", func() {
			So(auth.CanRejectInvitation(ctx, invitationID, ""), ShouldBeFalse)
		})

		Convey("Invalid invitationID should return false", func() {
			callerID := util.NewUserID()
			invitationID := testutil.MustNewID()
			authHelpers.On("IsAdmin", callerID).Return(false)

			Convey("Nil invitation", func() {
				backend.On("GetInvitationByID", mock.Anything, invitationID).Return(nil, nil)
				So(auth.CanRejectInvitation(ctx, invitationID, callerID), ShouldBeFalse)
			})

			Convey("Error getting invitation", func() {
				backend.On("GetInvitationByID", mock.Anything, invitationID).Return(nil, errors.New("test error"))
				So(auth.CanRejectInvitation(ctx, invitationID, callerID), ShouldBeFalse)
			})
		})

		Convey("Admins as callerID should return true", func() {
			adminUserID := util.NewUserID()

			authHelpers.On("IsTwitchAdmin", mock.Anything, adminUserID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, adminUserID, recipientID).Return(false, nil)
			authHelpers.On("IsInvitationRecipient", mock.Anything, invitation).Return(false)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanRejectInvitation(ctx, invitationID, adminUserID), ShouldBeTrue)
		})

		Convey("Twitch Admins as callerID should return true", func() {
			twitchAdminID := util.NewUserID()

			authHelpers.On("IsAdmin", twitchAdminID).Return(false)
			authHelpers.On("IsTwitchEditor", mock.Anything, twitchAdminID, recipientID).Return(false, nil)
			authHelpers.On("IsInvitationRecipient", twitchAdminID, invitation).Return(false)

			authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
			So(auth.CanRejectInvitation(ctx, invitationID, twitchAdminID), ShouldBeTrue)
		})

		Convey("Editors for Invitation Recipient as callerID should return true", func() {
			editorID := util.NewUserID()

			authHelpers.On("IsAdmin", editorID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, editorID).Return(false, nil)
			authHelpers.On("IsInvitationRecipient", editorID, invitation).Return(false)

			authHelpers.On("IsTwitchEditor", mock.Anything, editorID, recipientID).Return(true, nil)
			So(auth.CanRejectInvitation(ctx, invitationID, editorID), ShouldBeTrue)
		})

		Convey("Invitation Recipient as callerID should return true", func() {
			authHelpers.On("IsAdmin", recipientID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, recipientID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, recipientID, recipientID).Return(false, nil)

			authHelpers.On("IsInvitationRecipient", recipientID, invitation).Return(true)
			So(auth.CanRejectInvitation(ctx, invitationID, recipientID), ShouldBeTrue)
		})

		Convey("Unauthorized Users should return false", func() {
			userID := util.NewUserID()
			authHelpers.On("IsAdmin", userID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, userID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, userID, recipientID).Return(false, nil)
			authHelpers.On("IsInvitationRecipient", userID, invitation).Return(false)

			So(auth.CanRejectInvitation(ctx, invitationID, userID), ShouldBeFalse)
		})
	})
}
