package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// RejectOutOfNetworkInvitations wraps the corresponding backend function with an authorizer
func (a *auth) RejectOutOfNetworkInvitations(ctx context.Context, channelID, callerID string, reasonRejected models.InvitationReasonRejected) error {
	if !a.authorizer.CanRejectOutOfNetworkInvitations(ctx, channelID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &channelID,
			Method:          models.ErrorMethodTypeRejectOutOfNetworkInvitations,
			ErrorCode:       meepoerrors.ErrUnauthorized,
		})
		return twirp.NewError(twirp.PermissionDenied, "The caller cannot reject the channel's out of network invitations").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.RejectOutOfNetworkInvitations(ctx, channelID, callerID, reasonRejected)
}

// CanRejectOutOfNetworkInvitations determines whether all of a channel's out of network invitations.
func (a *auth) CanRejectOutOfNetworkInvitations(ctx context.Context, channelID, callerID string) bool {
	if callerID == "" || channelID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	if channelID == callerID {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchEditor(ctx, callerID, channelID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
