package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// RemoveMember wraps the corresponding backend function with a authorizer
func (a *auth) RemoveMember(ctx context.Context, memberID, squadID, callerID string) (*models.Squad, error) {
	if !a.authorizer.CanRemoveMember(ctx, memberID, squadID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			SquadID:         squadID,
			ChannelID:       callerID,
			TargetChannelID: &memberID,
			Method:          models.ErrorMethodTypeRemoveMember,
			ErrorCode:       meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot remove the member").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.RemoveMember(ctx, memberID, squadID, callerID)
}

// CanRemoveMember determines whether a callerID is allowed remove a member from squad
func (a *auth) CanRemoveMember(ctx context.Context, memberID, squadID, callerID string) bool {
	if callerID == "" || memberID == "" || squadID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	squad, err := a.backend.GetSquadByID(ctx, squadID)
	if err != nil || squad == nil {
		a.log.LogCtx(ctx, "err", err, "could not get squad")
		return false
	}

	if squad.OwnerID == nil {
		return false
	}

	if *squad.OwnerID == callerID {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchEditor(ctx, callerID, *squad.OwnerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
