package auth_test

import (
	"context"
	"testing"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"

	"github.com/pkg/errors"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanCanRemoveMember(t *testing.T) {
	Convey("CanRemoveMember", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)
		backend := (i.Backend).(*mocks.Backender)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		ownerID := util.NewUserID()
		squadID := testutil.MustNewID()
		squad := &models.Squad{
			ID:      squadID,
			OwnerID: &ownerID,
		}
		backend.On("FireSquadStreamErrorTrackingEvent", mock.Anything, mock.Anything)
		backend.On("GetSquadByID", mock.Anything, squadID).Return(squad, nil)

		Convey("No memberID should return false", func() {
			So(auth.CanRemoveMember(ctx, "", squadID, util.NewUserID()), ShouldBeFalse)
		})

		Convey("No squadID should return false", func() {
			So(auth.CanRemoveMember(ctx, util.NewUserID(), "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No callerID should return false", func() {
			So(auth.CanRemoveMember(ctx, util.NewUserID(), squadID, ""), ShouldBeFalse)
		})

		Convey("Invalid squadIDs should return false", func() {
			squadID := testutil.MustNewID()
			callerID := util.NewUserID()
			authHelpers.On("IsAdmin", callerID).Return(false)

			Convey("Non-existent squad", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(nil, nil)
				So(auth.CanRemoveMember(ctx, util.NewUserID(), squadID, callerID), ShouldBeFalse)
			})

			Convey("Nil ownerID", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(&models.Squad{
					ID:      squadID,
					OwnerID: nil,
				}, nil)
				So(auth.CanRemoveMember(ctx, util.NewUserID(), squadID, callerID), ShouldBeFalse)
			})

			Convey("Error getting squads", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(nil, errors.New("test error"))
				So(auth.CanRemoveMember(ctx, util.NewUserID(), squadID, callerID), ShouldBeFalse)
			})
		})

		Convey("Admins as callerID should return true", func() {
			adminUserID := util.NewUserID()

			authHelpers.On("IsTwitchAdmin", mock.Anything, adminUserID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, adminUserID, ownerID).Return(false, nil)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanRemoveMember(ctx, util.NewUserID(), squadID, adminUserID), ShouldBeTrue)
		})

		Convey("Twitch Admins as callerID should return true", func() {
			twitchAdminID := util.NewUserID()

			authHelpers.On("IsAdmin", twitchAdminID).Return(false)
			authHelpers.On("IsTwitchEditor", mock.Anything, twitchAdminID, ownerID).Return(false, nil)

			authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
			So(auth.CanRemoveMember(ctx, util.NewUserID(), squadID, twitchAdminID), ShouldBeTrue)
		})

		Convey("Twitch Editors as callerID should return true", func() {
			editorID := util.NewUserID()

			authHelpers.On("IsAdmin", editorID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, editorID).Return(false, nil)

			authHelpers.On("IsTwitchEditor", mock.Anything, editorID, ownerID).Return(true, nil)
			So(auth.CanRemoveMember(ctx, util.NewUserID(), squadID, editorID), ShouldBeTrue)
		})

		Convey("Squad OwnerID as callerID should return true", func() {
			authHelpers.On("IsAdmin", ownerID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, ownerID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, ownerID, ownerID).Return(false, nil)

			So(auth.CanRemoveMember(ctx, util.NewUserID(), squadID, ownerID), ShouldBeTrue)
		})

		Convey("Unauthorized callerID should return false", func() {
			userID := util.NewUserID()
			authHelpers.On("IsAdmin", userID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, userID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, userID, ownerID).Return(false, nil)

			So(auth.CanRemoveMember(ctx, util.NewUserID(), squadID, userID), ShouldBeFalse)
		})
	})
}
