package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// UpdateSquad wraps the corresponding backend function with an authorizer
func (a *auth) UpdateSquad(ctx context.Context, squadID string, status models.SquadStatus, callerID string) (*models.Squad, error) {
	if !a.authorizer.CanUpdateSquad(ctx, squadID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			SquadID:   squadID,
			Method:    models.ErrorMethodTypeUpdateSquad,
			ErrorCode: meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot update squad status").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.UpdateSquad(ctx, squadID, status, callerID)
}

// CanUpdateSquad determines whether the caller is authorized to a update squad status.
func (a *auth) CanUpdateSquad(ctx context.Context, squadID string, callerID string) bool {
	if callerID == "" || squadID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	squad, err := a.backend.GetSquadByID(ctx, squadID)
	if err != nil || squad == nil {
		return false
	}

	if squad.OwnerID == nil {
		return false
	}

	if *squad.OwnerID == callerID {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchEditor(ctx, callerID, *squad.OwnerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
