package auth_test

import (
	"context"
	"testing"

	"github.com/pkg/errors"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanUpdateSquad(t *testing.T) {
	Convey("CanUpdateSquad", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)
		backend := (i.Backend).(*mocks.Backender)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		squadID := testutil.MustNewID()
		ownerID := util.NewUserID()
		backend.On("GetSquadByID", mock.Anything, squadID).Return(&models.Squad{
			ID:      squadID,
			OwnerID: &ownerID,
		}, nil)

		Convey("No squadID should return false", func() {
			So(auth.CanUpdateSquad(ctx, "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No callerID should return false", func() {
			So(auth.CanUpdateSquad(ctx, testutil.MustNewID(), ""), ShouldBeFalse)
		})

		Convey("Invalid squadID should return false", func() {
			squadID := testutil.MustNewID()
			callerID := util.NewUserID()

			Convey("Non-existent squads should return false", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(nil, nil)
				authHelpers.On("IsAdmin", callerID).Return(false)

				So(auth.CanUpdateSquad(ctx, squadID, callerID), ShouldBeFalse)
			})

			Convey("No ownerID should return false", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(&models.Squad{
					ID:      squadID,
					OwnerID: nil,
				}, nil)
				authHelpers.On("IsAdmin", callerID).Return(false)

				So(auth.CanUpdateSquad(ctx, squadID, callerID), ShouldBeFalse)
			})

			Convey("Error getting squads should return false", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(nil, errors.New("test error"))
				authHelpers.On("IsAdmin", callerID).Return(false)

				So(auth.CanUpdateSquad(ctx, squadID, callerID), ShouldBeFalse)
			})
		})

		Convey("Admins as callerID should return true", func() {
			adminUserID := util.NewUserID()
			authHelpers.On("IsTwitchEditor", mock.Anything, adminUserID, ownerID).Return(false, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, adminUserID).Return(false, nil)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanUpdateSquad(ctx, squadID, adminUserID), ShouldBeTrue)
		})

		Convey("Twitch Admins as callerID should return true", func() {
			twitchAdminID := util.NewUserID()
			authHelpers.On("IsAdmin", twitchAdminID).Return(false)
			authHelpers.On("IsTwitchEditor", mock.Anything, twitchAdminID, ownerID).Return(false, nil)

			authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
			So(auth.CanUpdateSquad(ctx, squadID, twitchAdminID), ShouldBeTrue)
		})

		Convey("Squad Owners as callerID should return true", func() {
			authHelpers.On("IsAdmin", ownerID).Return(false)
			authHelpers.On("IsTwitchEditor", mock.Anything, ownerID, ownerID).Return(false, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, ownerID).Return(false, nil)

			So(auth.CanUpdateSquad(ctx, squadID, ownerID), ShouldBeTrue)
		})

		Convey("Unauthorized user as callerID should return false", func() {
			callerID := util.NewUserID()
			authHelpers.On("IsAdmin", callerID).Return(false)
			authHelpers.On("IsTwitchEditor", mock.Anything, callerID, ownerID).Return(false, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, callerID).Return(false, nil)

			So(auth.CanUpdateSquad(ctx, squadID, callerID), ShouldBeFalse)
		})
	})
}
