package auth

import (
	"context"
	"strings"

	"code.justin.tv/feeds/log"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// Utils contains auth helper functions used to determine a userID's authorization
type Utils interface {
	IsInvitationSender(userID string, invitation *models.DBInvitation) bool
	IsInvitationRecipient(userID string, invitation *models.DBInvitation) bool
	IsAdmin(userID string) bool
	IsTwitchAdmin(ctx context.Context, userID string) (bool, error)
	IsTwitchEditor(ctx context.Context, editorID, channelID string) (bool, error)
	IsTwitchEditorForSquadMember(ctx context.Context, squad *models.Squad, editorID string) bool
	IsSquadMember(squad *models.Squad, channelID string) bool
}

type utils struct {
	config  *Config
	clients *Clients
	log     *log.ElevatedLog
}

// NewUtils initializes a new Utils to be used for tests
func NewUtils(config *Config, clients *Clients, logger *log.ElevatedLog) Utils {
	return &utils{
		config:  config,
		clients: clients,
		log:     logger,
	}
}

// isInvitationRecipient checks whether a user is the recipient of an invitation.
func (u *utils) IsInvitationRecipient(userID string, invitation *models.DBInvitation) bool {
	if userID == "" || invitation == nil {
		return false
	}

	return invitation.RecipientID == userID
}

// IsInvitationSender checks whether a user is the sender of an invitation.
func (u *utils) IsInvitationSender(userID string, invitation *models.DBInvitation) bool {
	if userID == "" || invitation == nil {
		return false
	}

	return invitation.SenderID == userID
}

// isAdmin checking whether a user is allowed to perform any operation in the meepo
// system using a list of user IDs in Consul.
// Edit: Consul use is now deprecated and no longer called by Meepo. Other distconf
// methods may work for populating a manual admin users list, so leaving this function.
func (u *utils) IsAdmin(userID string) bool {
	if userID == "" {
		return false
	}

	adminsStr := u.config.adminUsers.Get()
	admins := strings.Split(adminsStr, ",")
	for _, admin := range admins {
		if admin == userID {
			return true
		}
	}
	return false
}

// isTwitchAdmin checks whether a user is a Twitch admin.
func (u *utils) IsTwitchAdmin(ctx context.Context, userID string) (bool, error) {
	if userID == "" {
		return false, nil
	}

	user, err := u.clients.Users.GetUserByID(ctx, userID)
	if err != nil || user == nil || user.Subadmin == nil {
		return false, err
	}

	return *user.Subadmin, nil
}

// IsTwitchEditor checks whether a user is an editor of channelID.
func (u *utils) IsTwitchEditor(ctx context.Context, editorID, channelID string) (bool, error) {
	if channelID == "" || editorID == "" {
		return false, nil
	}

	isEditor, err := u.clients.Hallpass.IsEditor(ctx, channelID, editorID)
	if err != nil {
		return false, err
	}

	return isEditor, nil
}

// IsTwitchEditorForSquadMember checks whether a user is an editor of a squad member.
func (u *utils) IsTwitchEditorForSquadMember(ctx context.Context, squad *models.Squad, editorID string) bool {
	if squad == nil || editorID == "" {
		return false
	}

	hasEditor := make(chan bool, len(squad.MemberIDs))

	for _, member := range squad.MemberIDs {
		go func(m string) {
			isEditor, err := u.IsTwitchEditor(ctx, editorID, m)
			if err != nil {
				u.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
				hasEditor <- false
				return
			}

			if !isEditor {
				hasEditor <- false
				return
			}

			hasEditor <- true
		}(member)
	}

	for i := 0; i < len(squad.MemberIDs); i++ {
		if resp := <-hasEditor; resp {
			return true
		}
	}

	return false
}

// IsSquadMember determines whether a callerID is a member of the squad
func (u *utils) IsSquadMember(squad *models.Squad, channelID string) bool {
	if squad == nil || channelID == "" {
		return false
	}

	for _, member := range squad.MemberIDs {
		if member == channelID {
			return true
		}
	}

	return false
}
