package auth_test

import (
	"context"
	"errors"
	"strings"
	"testing"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"
	usermodels "code.justin.tv/web/users-service/models"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestIsInvitationRecipient(t *testing.T) {
	i := setUpInjectables()
	authConfig, err := SetUpAuthConfig()
	if err != nil {
		t.Error("Unable to setup Auth configs")
	}
	helpers := SetUpForUtils(t, authConfig, i)
	if err != nil {
		t.Error("Unable to setup Helpers")
	}

	Convey("No userID should return false", t, func() {
		So(helpers.IsInvitationRecipient("", &models.DBInvitation{ID: testutil.MustNewID(), RecipientID: util.NewUserID()}), ShouldBeFalse)
	})

	Convey("No invitation should return false", t, func() {
		So(helpers.IsInvitationRecipient(util.NewUserID(), nil), ShouldBeFalse)
	})

	Convey("UserID different than RecipientID should return false", t, func() {
		So(helpers.IsInvitationRecipient(util.NewUserID(), &models.DBInvitation{ID: testutil.MustNewID(), RecipientID: util.NewUserID()}), ShouldBeFalse)
		So(helpers.IsInvitationRecipient(util.NewUserID(), &models.DBInvitation{ID: testutil.MustNewID(), RecipientID: ""}), ShouldBeFalse)
	})

	Convey("UserID being the same as RecipientID should return true", t, func() {
		recipientID := util.NewUserID()
		So(helpers.IsInvitationRecipient(recipientID, &models.DBInvitation{ID: testutil.MustNewID(), RecipientID: recipientID}), ShouldBeTrue)
	})
}

func TestIsInvitationSender(t *testing.T) {
	i := setUpInjectables()
	authConfig, err := SetUpAuthConfig()
	if err != nil {
		t.Error("Unable to setup Auth configs")
	}
	helpers := SetUpForUtils(t, authConfig, i)
	if err != nil {
		t.Error("Unable to setup Helpers")
	}

	Convey("No userID should return false", t, func() {
		So(helpers.IsInvitationSender("", &models.DBInvitation{ID: testutil.MustNewID(), SenderID: util.NewUserID()}), ShouldBeFalse)
	})

	Convey("No invitation should return false", t, func() {
		So(helpers.IsInvitationSender(util.NewUserID(), nil), ShouldBeFalse)
	})

	Convey("UserID different than SenderID should return false", t, func() {
		So(helpers.IsInvitationSender(util.NewUserID(), &models.DBInvitation{ID: testutil.MustNewID(), SenderID: util.NewUserID()}), ShouldBeFalse)
		So(helpers.IsInvitationSender(util.NewUserID(), &models.DBInvitation{ID: testutil.MustNewID(), SenderID: ""}), ShouldBeFalse)
	})

	Convey("UserID being the same as SenderID should return true", t, func() {
		senderID := util.NewUserID()
		So(helpers.IsInvitationSender(senderID, &models.DBInvitation{ID: testutil.MustNewID(), SenderID: senderID}), ShouldBeTrue)
	})
}

func TestIsAdmin(t *testing.T) {
	adminUserIDs := []string{util.NewUserID(), util.NewUserID()}

	i := setUpInjectables()
	authConfig, err := SetUpAuthConfig(map[string][]byte{
		"meepo.admin_users": []byte(strings.Join(adminUserIDs, ",")),
	})
	if err != nil {
		t.Error("Unable to setup Auth configs")
	}
	helpers := SetUpForUtils(t, authConfig, i)
	if err != nil {
		t.Error("Unable to setup Helpers")
	}

	Convey("No userID should return false", t, func() {
		So(helpers.IsAdmin(""), ShouldBeFalse)
	})

	Convey("UserID different than AdminUsers should return false", t, func() {
		So(helpers.IsAdmin(util.NewUserID()), ShouldBeFalse)
	})

	Convey("UserID in list of AdminUsers should return true", t, func() {
		So(helpers.IsAdmin(adminUserIDs[0]), ShouldBeTrue)
		So(helpers.IsAdmin(adminUserIDs[1]), ShouldBeTrue)
	})
}

func TestIsTwitchAdmin(t *testing.T) {
	Convey("IsTwitchAdmin", t, func() {
		i := setUpInjectables()
		usersClient := (i.Users).(*mocks.UsersClient)

		authConfig, err := SetUpAuthConfig()
		if err != nil {
			t.Error("Unable to setup Auth configs")
		}
		helpers := SetUpForUtils(t, authConfig, i)
		if err != nil {
			t.Error("Unable to setup Helpers")
		}
		ctx := context.Background()

		Convey("No userID should return false", func() {
			subAdmin := true
			usersClient.On("GetUserByID", mock.Anything, mock.Anything).Return(&usermodels.Properties{
				ID:       util.NewUserID(),
				Subadmin: &subAdmin,
			}, nil)

			isAdmin, err := helpers.IsTwitchAdmin(ctx, "")
			So(isAdmin, ShouldBeFalse)
			So(err, ShouldBeNil)
		})

		Convey("Errors UsersClient should return false and error", func() {
			mockedError := errors.New("mocked error")
			userID := util.NewUserID()
			usersClient.On("GetUserByID", mock.Anything, userID).Return(nil, mockedError)

			isAdmin, err := helpers.IsTwitchAdmin(ctx, userID)
			So(isAdmin, ShouldBeFalse)
			So(err, ShouldNotBeNil)
			So(err, ShouldEqual, mockedError)
		})

		Convey("Nil user result from UsersClient should return false", func() {
			userID := util.NewUserID()
			usersClient.On("GetUserByID", mock.Anything, userID).Return(nil, nil)

			isAdmin, err := helpers.IsTwitchAdmin(ctx, userID)
			So(isAdmin, ShouldBeFalse)
			So(err, ShouldBeNil)
		})

		Convey("User result from UsersClient with nil SubAdmin field should return false", func() {
			userID := util.NewUserID()
			usersClient.On("GetUserByID", mock.Anything, userID).Return(&usermodels.Properties{
				ID:       util.NewUserID(),
				Subadmin: nil,
			}, nil)

			isAdmin, err := helpers.IsTwitchAdmin(ctx, userID)
			So(isAdmin, ShouldBeFalse)
			So(err, ShouldBeNil)
		})

		Convey("Subadmin user result from UsersClient should return true", func() {
			userID := util.NewUserID()
			subAdmin := true
			usersClient.On("GetUserByID", mock.Anything, userID).Return(&usermodels.Properties{
				ID:       util.NewUserID(),
				Subadmin: &subAdmin,
			}, nil)

			isAdmin, err := helpers.IsTwitchAdmin(ctx, userID)
			So(isAdmin, ShouldBeTrue)
			So(err, ShouldBeNil)
		})
	})
}

func TestIsTwitchEditor(t *testing.T) {
	Convey("IsTwitchEditor", t, func() {
		i := setUpInjectables()
		hallpassClient := (i.Hallpass).(*mocks.HallpassClient)

		authConfig, err := SetUpAuthConfig()
		if err != nil {
			t.Error("Unable to setup Auth configs")
		}
		helpers := SetUpForUtils(t, authConfig, i)
		if err != nil {
			t.Error("Unable to setup Helpers")
		}
		ctx := context.Background()

		Convey("No editorID should return false", func() {
			hallpassClient.On("IsEditor", mock.Anything, mock.Anything, mock.Anything).Return(true, nil)

			isEditor, err := helpers.IsTwitchEditor(ctx, "", util.NewUserID())
			So(isEditor, ShouldBeFalse)
			So(err, ShouldBeNil)
		})

		Convey("No channelID should return false", func() {
			hallpassClient.On("IsEditor", mock.Anything, mock.Anything, mock.Anything).Return(true, nil)

			isEditor, err := helpers.IsTwitchEditor(ctx, util.NewUserID(), "")
			So(isEditor, ShouldBeFalse)
			So(err, ShouldBeNil)
		})

		Convey("Error from HallpassClient should return false and error", func() {
			mockedError := errors.New("mocked error")
			editorID := util.NewUserID()
			channelID := util.NewUserID()
			hallpassClient.On("IsEditor", mock.Anything, channelID, editorID).Return(false, mockedError)

			isEditor, err := helpers.IsTwitchEditor(ctx, editorID, channelID)
			So(isEditor, ShouldBeFalse)
			So(err, ShouldNotBeNil)
			So(err, ShouldEqual, mockedError)
		})

		Convey("Editors for ChannelID should return true", func() {
			editorID := util.NewUserID()
			channelID := util.NewUserID()
			hallpassClient.On("IsEditor", mock.Anything, channelID, editorID).Return(true, nil)

			isEditor, err := helpers.IsTwitchEditor(ctx, editorID, channelID)
			So(isEditor, ShouldBeTrue)
			So(err, ShouldBeNil)
		})
	})
}

func TestIsTwitchEditorForSquadMember(t *testing.T) {
	Convey("IsTwitchEditorForSquadMember", t, func() {
		i := setUpInjectables()
		hallpassClient := (i.Hallpass).(*mocks.HallpassClient)

		authConfig, err := SetUpAuthConfig()
		if err != nil {
			t.Error("Unable to setup Auth configs")
		}
		helpers := SetUpForUtils(t, authConfig, i)
		if err != nil {
			t.Error("Unable to setup Helpers")
		}

		ctx := context.Background()

		Convey("No squad should return false", func() {
			So(helpers.IsTwitchEditorForSquadMember(ctx, nil, util.NewUserID()), ShouldBeFalse)
		})

		Convey("No editorID should return false", func() {
			squad := models.Squad{
				MemberIDs: []string{util.NewUserID(), util.NewUserID()},
			}
			So(helpers.IsTwitchEditorForSquadMember(ctx, &squad, ""), ShouldBeFalse)
		})

		Convey("Squad without members should return false", func() {
			hallpassClient.On("IsEditor", mock.Anything, mock.Anything, mock.Anything).Return(true, nil)

			squad := models.Squad{
				MemberIDs: []string{},
			}
			So(helpers.IsTwitchEditorForSquadMember(ctx, &squad, util.NewUserID()), ShouldBeFalse)
		})

		Convey("ChannelID not an editor of any squad member should return false", func() {
			channelID := util.NewUserID()
			editorID := util.NewUserID()
			memberID1 := util.NewUserID()
			memberID2 := util.NewUserID()
			hallpassClient.On("IsEditor", mock.Anything, channelID, editorID).Return(false, nil)
			hallpassClient.On("IsEditor", mock.Anything, memberID1, editorID).Return(false, nil)
			hallpassClient.On("IsEditor", mock.Anything, memberID2, editorID).Return(false, nil)

			squad := models.Squad{
				MemberIDs: []string{memberID1, memberID2, channelID},
			}
			So(helpers.IsTwitchEditorForSquadMember(ctx, &squad, editorID), ShouldBeFalse)
		})

		Convey("Editor as ChannelID should return true", func() {
			channelID := util.NewUserID()
			editorID := util.NewUserID()
			memberID1 := util.NewUserID()
			memberID2 := util.NewUserID()
			hallpassClient.On("IsEditor", mock.Anything, channelID, editorID).Return(true, nil)
			hallpassClient.On("IsEditor", mock.Anything, memberID1, editorID).Return(false, nil)
			hallpassClient.On("IsEditor", mock.Anything, memberID2, editorID).Return(false, nil)

			squad := models.Squad{
				MemberIDs: []string{memberID1, memberID2, channelID},
			}
			So(helpers.IsTwitchEditorForSquadMember(ctx, &squad, editorID), ShouldBeTrue)
		})
	})
}

func TestIsSquadMember(t *testing.T) {
	Convey("IsSquadMember", t, func() {
		i := setUpInjectables()

		authConfig, err := SetUpAuthConfig()
		if err != nil {
			t.Error("Unable to setup Auth configs")
		}
		helpers := SetUpForUtils(t, authConfig, i)
		if err != nil {
			t.Error("Unable to setup Helpers")
		}

		Convey("No squad should return false", func() {
			So(helpers.IsSquadMember(nil, util.NewUserID()), ShouldBeFalse)
		})

		Convey("No channelID should return false", func() {
			squad := models.Squad{
				MemberIDs: []string{util.NewUserID(), util.NewUserID()},
			}
			So(helpers.IsSquadMember(&squad, ""), ShouldBeFalse)
		})

		Convey("Squad without members should return false", func() {
			squad := models.Squad{
				MemberIDs: []string{},
			}
			So(helpers.IsSquadMember(&squad, util.NewUserID()), ShouldBeFalse)
		})

		Convey("ChannelID not a squad member should return false", func() {
			squad := models.Squad{
				MemberIDs: []string{util.NewUserID(), util.NewUserID()},
			}
			So(helpers.IsSquadMember(&squad, util.NewUserID()), ShouldBeFalse)
		})

		Convey("Squad member as ChannelID should return true", func() {
			channelID := util.NewUserID()
			squad := models.Squad{
				MemberIDs: []string{channelID, util.NewUserID()},
			}
			So(helpers.IsSquadMember(&squad, channelID), ShouldBeTrue)
		})
	})
}
