package backend

import (
	"context"
	"fmt"

	"code.justin.tv/feeds/errors"
	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// AcceptInvitation accepts an invitation, removes the recipient from any squad they are in, and adds the recipient as a member to the squad.
func (b *backend) AcceptInvitation(ctx context.Context, invitationID, callerID string) (*models.Invitation, error) {
	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbInvitation, err := b.Datastore.GetInvitationByID(txCtx, invitationID)
	if err != nil {
		return nil, err
	}
	if dbInvitation == nil {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			InviteID:  invitationID,
			Method:    models.ErrorMethodTypeAcceptInvitation,
			ErrorCode: meepo_errors.ErrInvitationNotFound,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The invitation does not exist").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationNotFound)
	}
	if models.NewInvitationStatusFromDB(dbInvitation.Status) != models.InvitationStatusPending {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &dbInvitation.RecipientID,
			SquadID:         dbInvitation.SquadID,
			InviteID:        dbInvitation.ID,
			Method:          models.ErrorMethodTypeAcceptInvitation,
			ErrorCode:       meepo_errors.ErrInvitationCannotBeAccepted,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The invitation is not in pending state").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationCannotBeAccepted)
	}

	recipientID := dbInvitation.RecipientID
	squadID := dbInvitation.SquadID

	// Remove the recipient from their current squad if they are already part of a squad.
	var recipientOldSquad *models.ManagedSquad
	recipientDBSquad, err := b.Datastore.GetSquadByChannelID(txCtx, recipientID)
	if err != nil {
		return nil, err
	}
	var deletedInvitations []*models.DBInvitation
	if recipientDBSquad != nil {
		recipientOldSquad, err = b.handleDeleteMemberDB(txCtx, recipientID, recipientDBSquad)
		if err != nil {
			return nil, err
		}

		// Delete all existing invitations from the squad if owner has been removed
		deletedInvitations, err = b.handleDeletedSquadOwner(txCtx, recipientID, recipientDBSquad)
		if err != nil {
			return nil, err
		}
	}

	// Update the invitation status.
	newDBInvitation, err := b.Datastore.UpdateInvitationStatus(txCtx, invitationID, models.InvitationStatusAccepted)
	if err != nil {
		return nil, err
	}
	if newDBInvitation == nil {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &dbInvitation.RecipientID,
			SquadID:         dbInvitation.SquadID,
			InviteID:        dbInvitation.ID,
			Method:          models.ErrorMethodTypeAcceptInvitation,
			ErrorCode:       meepo_errors.ErrInvitationNotFound,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The invitation does not exist").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationNotFound)
	}

	// Create the membership.
	createMemberInput := &models.CreateMemberInput{
		SquadID:  squadID,
		MemberID: recipientID,
		Status:   models.MemberStatusActive,
	}
	_, err = b.Datastore.CreateMember(txCtx, createMemberInput)
	if err != nil {
		return nil, err
	}

	// Load info we need for the responses
	squad, err := b.getManagedSquadByID(txCtx, squadID, nil)
	if err != nil {
		return nil, err
	} else if squad == nil {
		// Sanity check; since we remove all invitations on squad deletion, we should not reach here.
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &dbInvitation.RecipientID,
			SquadID:         dbInvitation.SquadID,
			InviteID:        dbInvitation.ID,
			Method:          models.ErrorMethodTypeAcceptInvitation,
			ErrorCode:       meepo_errors.ErrSquadNotFound,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, fmt.Sprintf("ManagedSquad with id %v does not exist", squadID)).WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrSquadNotFound)
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error accepting invitation")
	}

	invitation := models.NewInvitationFromDB(newDBInvitation)

	// Update cache entries for the recipient's old squad (if they had an old squad), and their current squad.
	b.cacheSquad(ctx, models.NewSquadFromManagedSquad(squad))
	b.cacheSquad(ctx, models.NewSquadFromManagedSquad(recipientOldSquad))

	// Publish updates to PubSub for the recipient's old squad (if they had an old squad), and their current squad.
	b.publishSquadToMembersAndSquad(ctx, squad)
	b.publishSquadToMembersAndSquad(ctx, recipientOldSquad)

	// Publish updated invitation status to recipient
	b.publishSquadInvitesToRecipient(ctx, recipientID)

	// Publish deleted invitation status if recipient was owner of old squad and removed
	b.publishSquadInvitesToRecipientFromUpdatedInvitations(ctx, deletedInvitations)

	// Publish updates on this user's live squad to SNS.
	if squad.Status == models.SquadStatusLive {
		b.Clients.ChannelStatePublisher.PublishChannelIsInLiveSquad(ctx, squadID, invitation.RecipientID)
	}

	var creatorAction models.CreatorActionType

	if squad.Status != models.SquadStatusLive {
		creatorAction = models.CreatorActionTypeAcceptInvite
	} else {
		creatorAction = models.CreatorActionTypeJoinSquad
	}

	b.fireCreatorActionTrackingEvent(ctx, models.CreatorActionTrackingEventSet{
		SquadID:      squad.ID,
		InvitationID: invitation.ID,
		MemberIDs:    squad.MemberIDs,
		OwnerID:      squad.OwnerID,
		SquadStatus:  &squad.Status,
		Events: []models.CreatorActionTrackingEventInfo{
			{
				ChannelID:     callerID,
				CreatorAction: creatorAction,
				CreatorMethod: models.CreatorMethodTypeClickAccept,
			},
		},
	})

	return invitation, nil
}
