package backend

import (
	"context"
	"time"

	"code.justin.tv/twitch-events/meepo/internal/clock"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/twitch-events/meepo/clients"
	"code.justin.tv/twitch-events/meepo/internal/cache"
	"code.justin.tv/twitch-events/meepo/internal/datastore"
	"code.justin.tv/twitch-events/meepo/internal/models"

	"code.justin.tv/eventbus/schema/pkg/user"
	"code.justin.tv/vod/eventbus"
	"github.com/golang/protobuf/ptypes/wrappers"
)

// Max limits for the meepo service.
const (
	// A squad can have at most 4 members and pending invitations together at the same time.
	MaxMembersAndPendingInvitations = 4
	MaxRejectedInvitationsPerSquad  = 3
	MaxPendingInvitationsPerChannel = 10
)

// Backender contains functionality that all Backend objects must support.
type Backender interface {
	AcceptInvitation(ctx context.Context, invitationID, callerID string) (*models.Invitation, error)
	DeleteInvitation(ctx context.Context, invitationID string, callerID string) (*models.Invitation, error)
	CanAccessSquads(ctx context.Context, channelID string) (bool, error)
	CreateInvitation(ctx context.Context, senderID, recipientID, callerID string) (*models.Invitation, error)
	CreateMembership(ctx context.Context, targetUserID, squadID, callerID string) (*models.Member, error)
	CreateSquad(ctx context.Context, ownerID, callerID string) (*models.Squad, error)
	DeleteUser(ctx context.Context, channelID string) error
	DestroyUser(ctx context.Context, channelID string) error
	GetPendingInvitationsCountByRecipientID(ctx context.Context, channelID string, callerID string) (int64, error)
	GetPendingInvitationsByRecipientID(ctx context.Context, channelID string) ([]*models.Invitation, error)
	GetInvitationByID(ctx context.Context, id string) (*models.DBInvitation, error)
	GetInvitationsBySquadID(ctx context.Context, squadID string, status models.InvitationStatus, callerID string) ([]*models.Invitation, error)
	GetInvitePolicyByChannelID(ctx context.Context, channelID, callerID string) (*models.InvitePolicy, error)
	GetSquadByChannelID(ctx context.Context, channelID string) (*models.Squad, error)
	GetSquadByID(ctx context.Context, id string) (*models.Squad, error)
	LeaveSquad(ctx context.Context, memberID, squadID, callerID string, wentOffline bool) (*models.Squad, error)
	LivecheckChannels(ctx context.Context) error
	RemoveMember(ctx context.Context, memberID, squadID, callerID string) (*models.Squad, error)
	RejectInvitation(ctx context.Context, invitationID, callerID string, reasonRejected models.InvitationReasonRejected) (*models.Invitation, error)
	RejectOutOfNetworkInvitations(ctx context.Context, channelID, callerID string, reasonRejected models.InvitationReasonRejected) error
	UpdateInvitePolicyByChannelID(ctx context.Context, channelID, callerID string, invitePolicy models.InvitationPolicy) (*models.InvitePolicy, error)
	UpdateSquad(ctx context.Context, squadID string, status models.SquadStatus, callerID string) (*models.Squad, error)

	ProcessUserDestroy(ctx context.Context, event *user.UserDestroy, cursor *wrappers.StringValue) (*eventbus.ConsumeEventResponse, error)

	DeleteExpiredData(ctx context.Context) (int64, error)

	FireSquadStreamErrorTrackingEvent(ctx context.Context, event models.SquadStreamErrorTrackingEventInfo)
}

// Config contains the configuration values for the backend.
type Config struct {
	adminUsers                   *distconf.Str
	alwaysOnChannels             *distconf.Str
	cacheDuration                *distconf.Duration
	cacheRefreshDuration         *distconf.Duration
	cacheRefreshProbability      *distconf.Float
	squadEnabledForAffiliates    *distconf.Bool
	squadEnabledForPartners      *distconf.Bool
	squadEnabledWhitelist        *distconf.Str
	loggingEnabled               *distconf.Bool
	livecheckNewSquadGracePeriod *distconf.Duration
	inviteNotificationWhitelist  *distconf.Str
	ttlInDays                    *distconf.Int

	// This is purely for testing and tells the backend to skip reading the cache.
	TestSkipCacheRead bool
}

// Load loads the config for the backend.
func (c *Config) Load(d *distconf.Distconf) error {
	c.adminUsers = d.Str("meepo.admin_users", "")
	c.alwaysOnChannels = d.Str("meepo.always_on_channels", "")
	c.cacheDuration = d.Duration("meepo.backend.cache_duration", 10*time.Minute)
	c.cacheRefreshDuration = d.Duration("meepo.backend.cache_refresh_duration", 6*time.Minute)
	c.cacheRefreshProbability = d.Float("meepo.backend.cache_refresh_probability", 0.001)
	c.squadEnabledForAffiliates = d.Bool("meepo.squad_enabled_for_affiliates", false)
	c.squadEnabledForPartners = d.Bool("meepo.squad_enabled_for_partners", false)
	c.squadEnabledWhitelist = d.Str("meepo.squad_enabled_whitelist", "")
	c.loggingEnabled = d.Bool("meepo.logging_enabled", false)
	c.livecheckNewSquadGracePeriod = d.Duration("meepo.livecheck_new_squad_grace_period", 10*time.Minute)
	c.inviteNotificationWhitelist = d.Str("meepo.invite_notification_whitelist", "")
	c.ttlInDays = d.Int("meepo.ttl_in_days", 28)
	return nil
}

// Clients contains the clients required for the backend.
type Clients struct {
	Follows               clients.FollowsClient
	Friendship            clients.FriendshipClient
	Liveline              clients.LivelineClient
	Pubsub                clients.PubsubClient
	Ripley                clients.RipleyClient
	Roster                clients.RosterClient
	Spade                 clients.SpadeClient
	Users                 clients.UsersClient
	ChannelStatePublisher clients.ChannelStatePublisher
	PDMS                  clients.PDMSClient
	Dart                  clients.NotifierClient
}

// backend contains meepo's datastore and clients to various services.
type backend struct {
	datastore.Datastore

	config *Config
	cache  *cache.Service
	*Clients

	log   *log.ElevatedLog
	stats *service_common.StatSender
	clock clock.Clock
}

var _ Backender = &backend{}

// NewBackend initializes a backend using the provided datastore.
func NewBackend(config *Config, datastore datastore.Datastore, clients *Clients, cache *cache.Service, logger *log.ElevatedLog, stats *service_common.StatSender, clock clock.Clock) (Backender, error) {
	return &backend{
		config:    config,
		Datastore: datastore,
		cache:     cache,
		Clients:   clients,
		log:       logger,
		stats:     stats,
		clock:     clock,
	}, nil
}
