package backend

import (
	"context"
	"encoding/json"
	"math/rand"
	"time"

	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// CachedSquad is a cached representation of a squad
// Update the cache key if you need to make a backwards incompatible change to Squad object
type cachedSquad struct {
	CachedAt time.Time `json:"ct"`

	ID        string             `json:"id"`
	MemberIDs []string           `json:"m_ids"`
	OwnerID   *string            `json:"o_id"`
	Status    models.SquadStatus `json:"s"`
	CreatedBy string             `json:"c_by"`

	CreatedAt time.Time `json:"c_at"`
	UpdatedAt time.Time `json:"u_at"`
}

func cachedSquadToSquad(cached *cachedSquad) *models.Squad {
	if cached == nil {
		return nil
	}

	return &models.Squad{
		ID:        cached.ID,
		MemberIDs: cached.MemberIDs,
		OwnerID:   cached.OwnerID,
		Status:    cached.Status,
		CreatedBy: cached.CreatedBy,

		CreatedAt: cached.CreatedAt,
		UpdatedAt: cached.UpdatedAt,
	}
}

func squadToCachedSquad(squad *models.Squad, cachedAt time.Time) *cachedSquad {
	if squad == nil {
		return nil
	}

	return &cachedSquad{
		CachedAt: cachedAt,

		ID:        squad.ID,
		MemberIDs: squad.MemberIDs,
		OwnerID:   squad.OwnerID,
		Status:    squad.Status,
		CreatedBy: squad.CreatedBy,

		CreatedAt: squad.CreatedAt,
		UpdatedAt: squad.UpdatedAt,
	}
}

func cachedSquadKey(squadID string) string {
	return "mpo:s1:" + squadID
}

func (b *backend) getCachedSquadByID(ctx context.Context, id string) *models.Squad {
	if b.config.TestSkipCacheRead {
		return nil
	}
	key := cachedSquadKey(id)
	cached, err := b.cache.Get(ctx, key)
	if err != nil {
		if err != memcache.ErrCacheMiss {
			b.log.LogCtx(ctx, "err", err, "key", key, "could not get squad from cache")
		}
		return nil
	}
	var squad cachedSquad
	err = json.Unmarshal(cached.Value, &squad)
	if err != nil {
		b.log.LogCtx(ctx, "err", err, "key", key, "could not unmarshall squad from cache")
		return nil
	}
	// This is an attempt to get past the thundering herd problem.
	// If the squad is on a busy channel and the cache expires, we may get a ton of requests to the database
	// at the same time.  This attempts to solve it by setting a probability to refresh the cache before the expiry.
	if time.Now().After(squad.CachedAt.Add(b.config.cacheRefreshDuration.Get())) &&
		rand.Float64() < b.config.cacheRefreshProbability.Get() {
		return nil
	}
	return cachedSquadToSquad(&squad)
}

func (b *backend) cacheSquad(ctx context.Context, squad *models.Squad) {
	if squad == nil {
		return
	}
	key := cachedSquadKey(squad.ID)
	data, err := json.Marshal(squadToCachedSquad(squad, time.Now()))
	if err != nil {
		b.log.LogCtx(ctx, "err", err, "key", key, "could not marshall squad for cache")
		return
	}
	err = b.cache.Set(ctx, &memcache.Item{
		Key:        key,
		Value:      data,
		Expiration: int32(b.config.cacheDuration.Get().Seconds()),
	})
	if err != nil {
		b.log.LogCtx(ctx, "err", err, "key", key, "could not cache squad")
		return
	}

	// This makes sure to update the channel mappings when we cache a squad
	membersSquadID := &squad.ID
	if squad.Status == models.SquadStatusEnded {
		// If the squad has ended, then we should record that the members are no longer
		// associated with a squad.
		membersSquadID = nil
	}
	b.cacheChannelMapping(ctx, membersSquadID, squad.MemberIDs)
}

func (b *backend) deleteSquadFromCache(ctx context.Context, squadID string) {
	if squadID == "" {
		return
	}
	key := cachedSquadKey(squadID)
	err := b.cache.Delete(ctx, key)
	if err != nil && err != memcache.ErrCacheMiss {
		b.log.LogCtx(ctx, "err", err, "key", key, "could not delete squad from cache")
	}
	return
}

func cachedChannelMappingKey(channelID string) string {
	return "mpo:c1:" + channelID
}

type cachedChannelMapping struct {
	CachedAt time.Time `json:"ct"`
	SquadID  *string   `json:"s_id"`
}

func (b *backend) getCachedChannelMapping(ctx context.Context, channelID string) (*string, bool) {
	if b.config.TestSkipCacheRead {
		return nil, false
	}
	key := cachedChannelMappingKey(channelID)
	cached, err := b.cache.Get(ctx, key)
	if err != nil {
		if err != memcache.ErrCacheMiss {
			b.log.LogCtx(ctx, "err", err, "key", key, "could not get channel mapping from cache")
		}
		return nil, false
	}
	var mapping cachedChannelMapping
	err = json.Unmarshal(cached.Value, &mapping)
	if err != nil {
		b.log.LogCtx(ctx, "err", err, "key", key, "could not unmarshall channel mapping from cache")
		return nil, false
	}
	// This is an attempt to get past the thundering herd problem.
	// If the squad is on a busy channel and the cache expires, we may get a ton of requests to the database
	// at the same time.  This attempts to solve it by setting a probability to refresh the cache before the expiry.
	if time.Now().After(mapping.CachedAt.Add(b.config.cacheRefreshDuration.Get())) &&
		rand.Float64() < b.config.cacheRefreshProbability.Get() {
		return nil, false
	}
	return mapping.SquadID, true
}

func (b *backend) cacheChannelMapping(ctx context.Context, squadID *string, channelIDs []string) {
	cachedAt := time.Now()
	for _, channelID := range channelIDs {
		key := cachedChannelMappingKey(channelID)
		data, err := json.Marshal(&cachedChannelMapping{
			CachedAt: cachedAt,
			SquadID:  squadID,
		})
		if err != nil {
			b.log.LogCtx(ctx, "err", err, "key", key, "could not marshall channel mapping for cache")
			continue
		}
		err = b.cache.Set(ctx, &memcache.Item{
			Key:        key,
			Value:      data,
			Expiration: int32(b.config.cacheDuration.Get().Seconds()),
		})
		if err != nil {
			b.log.LogCtx(ctx, "err", err, "key", key, "could not cache channel mapping")
			continue
		}
	}
}
