package backend

import (
	"context"

	"code.justin.tv/feeds/errors"
	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// CreateInvitation creates an invitation from a user's squad to another user, and creates the squad if user does not belong to a squad.
func (b *backend) CreateInvitation(ctx context.Context, senderID, recipientID, callerID string) (*models.Invitation, error) {
	if senderID == recipientID {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       senderID,
			TargetChannelID: &recipientID,
			Method:          models.ErrorMethodTypeCreateInvitation,
			ErrorCode:       meepo_errors.ErrInvitationInvalid,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The sender cannot be the same as the recipient").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationInvalid)
	}

	channelCanInviteUser, err := b.channelCanInviteUser(ctx, senderID, recipientID)
	if err != nil {
		return nil, err
	}
	if !channelCanInviteUser {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       senderID,
			TargetChannelID: &recipientID,
			Method:          models.ErrorMethodTypeCreateInvitation,
			ErrorCode:       meepo_errors.ErrInvitationBlocked,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The sender does not have permission to invite the recipient").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationBlocked)
	}

	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	var newSquadCreated bool
	squad, err := b.Datastore.GetSquadByChannelID(txCtx, senderID)
	if err != nil {
		return nil, err
	}
	if squad == nil {
		squad, err = b.createSquadWithTransaction(txCtx, senderID)
		if err != nil {
			return nil, err
		}
		newSquadCreated = true
	}

	rejectedInvitations, err := b.prepareToCreateInvitation(txCtx, squad, senderID, recipientID)
	if err != nil {
		return nil, err
	}
	rejectedManagedSquads, err := b.getManagedSquadsByIDs(txCtx, models.NewSquadIDsFromDBInvitations(rejectedInvitations))
	if err != nil {
		return nil, err
	}

	createInvitationInput := &models.CreatePendingInvitationInput{
		SquadID:     squad.ID,
		SenderID:    senderID,
		RecipientID: recipientID,
	}
	dbInvitation, err := b.Datastore.CreatePendingInvitation(txCtx, createInvitationInput)
	if err != nil {
		return nil, err
	}

	if dbInvitation == nil {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       senderID,
			TargetChannelID: &recipientID,
			Method:          models.ErrorMethodTypeCreateInvitation,
			ErrorCode:       meepo_errors.ErrInvitationAlreadyExists,
		})
		return nil, twirp.NewError(twirp.AlreadyExists, "The recipient is already invited to the squad").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationAlreadyExists)
	}

	invitation := models.NewInvitationFromDB(dbInvitation)

	managedSquad, err := b.getManagedSquadByID(txCtx, squad.ID, &preloadedSquadData{squad: squad})
	if err != nil {
		return nil, err
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving invitation")
	}

	b.publishSquadToMembersAndSquad(ctx, managedSquad)
	// Publish the updated squads to PubSub.  (This pubsub message messages will allow the squad owners'
	// widgets to render the rejected invitations.)
	b.publishSquadsToMembersAndSquad(ctx, rejectedManagedSquads)

	// Send pubsub message to recipient of invitation created
	b.publishSquadInvitesToRecipient(ctx, recipientID)
	// Send a pubsub message to the recipients with too many pending invitations
	b.publishSquadInvitesToRecipientFromUpdatedInvitations(ctx, rejectedInvitations)

	b.publishSquadStreamInviteDartNotification(ctx, senderID, recipientID, invitation.ID)

	if managedSquad != nil {
		b.fireCreatorActionTrackingEvent(ctx, models.CreatorActionTrackingEventSet{
			SquadID:      managedSquad.ID,
			InvitationID: invitation.ID,
			MemberIDs:    managedSquad.MemberIDs,
			OwnerID:      managedSquad.OwnerID,
			SquadStatus:  &managedSquad.Status,
			Events: []models.CreatorActionTrackingEventInfo{
				{
					ChannelID:     senderID,
					CreatorAction: models.CreatorActionTypeSendInvite,
				},
			},
		})

		if newSquadCreated {
			b.fireSquadStateChangeTrackingEvent(ctx, models.SquadStateChangeTrackingEventInfo{
				SquadID: managedSquad.ID,
				State:   models.SquadStatusPending,
				Method:  models.StateChangeMethodTypeFirstInviteSent,
			})
		}
	}

	return invitation, nil
}
