package backend

import (
	"context"
	"fmt"

	"code.justin.tv/feeds/errors"
	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// CreateMembership creates a membership of the target user to the squad.
func (b *backend) CreateMembership(ctx context.Context, targetUserID, squadID, callerID string) (*models.Member, error) {
	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbSquad, err := b.Datastore.GetSquadByID(txCtx, squadID)
	if err != nil {
		return nil, err
	}
	if dbSquad == nil {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &targetUserID,
			SquadID:         squadID,
			Method:          models.ErrorMethodTypeCreateMembership,
			ErrorCode:       meepo_errors.ErrSquadNotFound,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, fmt.Sprintf("Squad with id %v does not exist", squadID)).WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrSquadNotFound)
	}

	isFull, err := b.checkSquadAtCapacity(txCtx, dbSquad)
	if err != nil {
		return nil, err
	}
	if isFull {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &targetUserID,
			SquadID:         squadID,
			Method:          models.ErrorMethodTypeCreateMembership,
			ErrorCode:       meepo_errors.ErrSquadFull,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The squad is already full").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrSquadFull)
	}

	// Check if the target user is already in a squad.
	var targetUserSquad *models.ManagedSquad
	targetUserDBSquad, err := b.Datastore.GetSquadByChannelID(txCtx, targetUserID)
	if err != nil {
		return nil, err
	}
	var deletedInvitations []*models.DBInvitation
	if targetUserDBSquad != nil {
		if targetUserDBSquad.ID == squadID {
			b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
				ChannelID:       callerID,
				TargetChannelID: &targetUserID,
				SquadID:         squadID,
				Method:          models.ErrorMethodTypeCreateMembership,
				ErrorCode:       meepo_errors.ErrUserAlreadyInSquad,
			})
			return nil, twirp.NewError(twirp.InvalidArgument, "The recipient is already in the squad").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrUserAlreadyInSquad)
		}

		// Make the target user leave the old squad.
		targetUserSquad, err = b.handleDeleteMemberDB(txCtx, targetUserID, targetUserDBSquad)
		if err != nil {
			return nil, err
		}

		deletedInvitations, err = b.handleDeletedSquadOwner(txCtx, targetUserID, targetUserDBSquad)
		if err != nil {
			return nil, err
		}
	}

	// Create the membership.
	createMemberInput := &models.CreateMemberInput{
		SquadID:  squadID,
		MemberID: targetUserID,
		Status:   models.MemberStatusActive,
	}
	dbMember, err := b.Datastore.CreateMember(txCtx, createMemberInput)
	if err != nil {
		return nil, err
	}

	squadAfter, err := b.getManagedSquadByID(txCtx, squadID, &preloadedSquadData{squad: dbSquad})
	if err != nil {
		return nil, err
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving membership")
	}

	// Update cache entries for the user's old squad (if they had an old squad), and their current squad.
	b.cacheSquad(ctx, models.NewSquadFromManagedSquad(targetUserSquad))
	b.cacheSquad(ctx, models.NewSquadFromManagedSquad(squadAfter))

	// Publish updates to PubSub for the user's old squad (if they had an old squad), and their current squad.
	b.publishSquadToMembersAndSquad(ctx, targetUserSquad)
	b.publishSquadToMembersAndSquad(ctx, squadAfter)

	// Publish deleted invitation status if recipient was owner of old squad and removed
	b.publishSquadInvitesToRecipientFromUpdatedInvitations(ctx, deletedInvitations)

	// Publish an update to SNS that this user has become part of a live squad.
	if squadAfter.Status == models.SquadStatusLive {
		b.Clients.ChannelStatePublisher.PublishChannelIsInLiveSquad(ctx, squadID, targetUserID)
	}

	return models.NewMemberFromDB(dbMember), nil
}
