package backend

import (
	"context"

	"code.justin.tv/feeds/errors"
	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// CreateSquad creates a squad with the given owner.
func (b *backend) CreateSquad(ctx context.Context, ownerID, callerID string) (*models.Squad, error) {
	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	existingSquad, err := b.Datastore.GetSquadByChannelID(txCtx, ownerID)
	if err != nil {
		return nil, err
	}
	if existingSquad != nil {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &ownerID,
			SquadID:         existingSquad.ID,
			Method:          models.ErrorMethodTypeCreateSquad,
			ErrorCode:       meepo_errors.ErrUserInAnotherSquad,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The owner is already in a squad").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrUserInAnotherSquad)
	}

	squad, err := b.createSquadWithTransaction(txCtx, ownerID)
	if err != nil {
		return nil, err
	}
	squadID := squad.ID

	managedSquad, err := b.getManagedSquadByID(txCtx, squadID, &preloadedSquadData{squad: squad})
	if err != nil {
		return nil, err
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving membership")
	}

	squadAfter := models.NewSquadFromManagedSquad(managedSquad)
	b.cacheSquad(ctx, models.NewSquadFromManagedSquad(managedSquad))
	b.publishSquadToMembersAndSquad(ctx, managedSquad)

	return squadAfter, nil
}

func (b *backend) createSquadWithTransaction(txCtx context.Context, ownerID string) (*models.DBSquad, error) {
	createSquadInput := &models.CreateSquadInput{
		OwnerID:   ownerID,
		CreatedBy: ownerID,
		Status:    models.SquadStatusPending,
	}
	squad, err := b.Datastore.CreateSquad(txCtx, createSquadInput)
	if err != nil {
		return nil, err
	}

	// Add the user as a member of this squad.
	createMemberInput := &models.CreateMemberInput{
		SquadID:  squad.ID,
		MemberID: ownerID,
		Status:   models.MemberStatusActive,
	}
	_, err = b.Datastore.CreateMember(txCtx, createMemberInput)
	if err != nil {
		return nil, err
	}
	// TODO: need to send pubsub messages and update cache for the new squad

	return squad, nil
}
