package backend

import (
	"context"

	"code.justin.tv/feeds/errors"
	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// DeleteInvitation updates a pending or rejected invitation into deleted state, only allow squad owners to delete invitations.
func (b *backend) DeleteInvitation(ctx context.Context, invitationID string, callerID string) (*models.Invitation, error) {
	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbInvitation, err := b.Datastore.GetInvitationByID(txCtx, invitationID)
	if err != nil {
		return nil, err
	}
	if dbInvitation == nil {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			InviteID:  invitationID,
			Method:    models.ErrorMethodTypeDeleteInvitation,
			ErrorCode: meepo_errors.ErrInvitationNotFound,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The invitation does not exist").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationNotFound)
	}

	dbSquad, err := b.Datastore.GetSquadByID(txCtx, dbInvitation.SquadID)
	if err != nil {
		return nil, err
	}
	if dbSquad == nil {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &dbInvitation.RecipientID,
			InviteID:        invitationID,
			SquadID:         dbInvitation.SquadID,
			Method:          models.ErrorMethodTypeDeleteInvitation,
			ErrorCode:       meepo_errors.ErrSquadNotFound,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The squad does not exist").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrSquadNotFound)
	}

	// Check invitation status, can only delete PENDING or REJECTED invitation
	invitationStatus := models.NewInvitationStatusFromDB(dbInvitation.Status)
	if invitationStatus != models.InvitationStatusPending && invitationStatus != models.InvitationStatusRejected {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &dbInvitation.RecipientID,
			InviteID:        invitationID,
			SquadID:         dbInvitation.SquadID,
			Method:          models.ErrorMethodTypeDeleteInvitation,
			ErrorCode:       meepo_errors.ErrInvitationCannotBeDeleted,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The invitation is not in rejected nor pending state").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationCannotBeDeleted)
	}

	// Update status to deleted
	newDBInvitation, err := b.Datastore.UpdateInvitationStatus(txCtx, invitationID, models.InvitationStatusDeleted)
	if err != nil {
		return nil, err
	}
	if newDBInvitation == nil {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &dbInvitation.RecipientID,
			InviteID:        invitationID,
			SquadID:         dbInvitation.SquadID,
			Method:          models.ErrorMethodTypeDeleteInvitation,
			ErrorCode:       meepo_errors.ErrInvitationNotFound,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The invitation does not exist").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationNotFound)
	}

	squadID := newDBInvitation.SquadID
	// Load info we need for tracking
	members, err := b.Datastore.GetMembersBySquadID(txCtx, squadID)
	if err != nil {
		return nil, err
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error deleting invitation")
	}

	// Publish updated invitation status to recipient
	b.publishSquadInvitesToRecipient(ctx, dbInvitation.RecipientID)

	squad := models.NewManagedSquadFromDB(dbSquad, members, nil)

	b.fireCreatorActionTrackingEvent(ctx, models.CreatorActionTrackingEventSet{
		SquadID:      squadID,
		InvitationID: newDBInvitation.ID,
		MemberIDs:    squad.MemberIDs,
		OwnerID:      squad.OwnerID,
		SquadStatus:  &squad.Status,
		Events: []models.CreatorActionTrackingEventInfo{
			{
				ChannelID:     callerID,
				CreatorAction: models.CreatorActionTypeCancelInvite,
				CreatorMethod: models.CreatorMethodTypeOwnerClickCancel,
			},
		},
	})

	return models.NewInvitationFromDB(newDBInvitation), nil
}
