package backend

import (
	"context"

	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/pkg/errors"
	"github.com/twitchtv/twirp"
)

// handleDeleteMemberDB deletes a user's membership to a squad from the DB, and returns the updated squad.
// It does not do any auth check. The caller is expected to start and end the transaction.
func (b *backend) handleDeleteMemberDB(txCtx context.Context, memberID string, squad *models.DBSquad) (*models.ManagedSquad, error) {
	dbMember, err := b.Datastore.DeleteMember(txCtx, memberID)
	if err != nil {
		return nil, err
	}

	// Check if the user wasn't in a squad.
	if dbMember == nil {
		var squadID string
		if squad != nil {
			squadID = squad.ID
		}
		b.FireSquadStreamErrorTrackingEvent(txCtx, models.SquadStreamErrorTrackingEventInfo{
			TargetChannelID: &memberID,
			SquadID:         squadID,
			Method:          models.ErrorMethodTypeDeleteMember,
			ErrorCode:       meepo_errors.ErrUserNotInSquad,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The user is not in the squad").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrUserNotInSquad)
	}

	// Check if the user was in the squad specified.
	if dbMember.SquadID != squad.ID {
		var squadID string
		if squad != nil {
			squadID = squad.ID
		}
		b.FireSquadStreamErrorTrackingEvent(txCtx, models.SquadStreamErrorTrackingEventInfo{
			TargetChannelID: &memberID,
			SquadID:         squadID,
			Method:          models.ErrorMethodTypeDeleteMember,
			ErrorCode:       meepo_errors.ErrUserNotInSquad,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The user is not in the squad").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrUserNotInSquad)
	}

	squadID := squad.ID
	members, err := b.Datastore.GetMembersBySquadID(txCtx, squadID)
	if err != nil {
		return nil, err
	}

	var invitations []*models.DBInvitation

	if b.isOwnerWithSquad(memberID, squad) {
		// Find the earliest undeleted member to become the new owner
		if len(members) == 0 {
			// If there are no members left, end the squad
			squad, err = b.Datastore.EndSquad(txCtx, squad.ID)
			if err != nil {
				return nil, errors.Wrap(err, "error ending squad")
			}
		} else {
			squad, err = b.Datastore.UpdateSquadOwner(txCtx, squad.ID, members[0].MemberID)
			if err != nil {
				return nil, errors.Wrap(err, "error updating squad owner")
			}
		}
	} else {
		var err error
		invitations, err = b.loadOutgoingInvitations(txCtx, squadID)
		if err != nil {
			return nil, errors.Wrap(err, "error loading squad invitations")
		}
	}

	return models.NewManagedSquadFromDB(squad, members, invitations), nil
}

// handleDeletedSquadOwner takes a memberID that has been deleted and a previous squad object
// and determines whether the deleted member was the owner of the squad
// If the deleted member was the owner, delete all pending and rejected invitations from the squad
func (b *backend) handleDeletedSquadOwner(txCtx context.Context, deletedMemberID string, squad *models.DBSquad) ([]*models.DBInvitation, error) {
	if !b.isOwnerWithSquad(deletedMemberID, squad) {
		return nil, nil
	}

	// Delete all existing invitations from the squad
	deletedInvitations, err := b.Datastore.DeleteInvitationsBySquadID(
		txCtx,
		squad.ID,
		[]models.InvitationStatus{models.InvitationStatusPending, models.InvitationStatusRejected},
	)
	if err != nil {
		return nil, errors.Wrap(err, "error deleting squad invitations")
	}

	return deletedInvitations, nil
}

// After a member gets removed from a squad, deleteMemberFromCacheAndPublishUpdates:
// - updates the cache
// - sends messages to pubsub for the updated squad, each member in the squad, and the removed member
// - sends a message to SNS indicating that the member is no longer part of a live squad.
func (b *backend) deleteMemberFromCacheAndPublishUpdates(ctx context.Context, deletedMemberID string, squad *models.ManagedSquad) {
	// Update the cache with the updated squad, and that the member being removed is not
	// associated with any squad.  (We could invalidate the squad in the cache, but that could
	// cause thundering herd issues.)
	b.cacheSquad(ctx, models.NewSquadFromManagedSquad(squad))
	b.cacheChannelMapping(ctx, nil, []string{deletedMemberID})

	b.publishSquadToMembersAndSquad(ctx, squad)
	err := b.Pubsub.PublishChannelNotInSquad(ctx, deletedMemberID)
	if err != nil {
		b.log.LogCtx(ctx, "squad_id", nil, "member_id", deletedMemberID, "err", err, "error sending pubsub message")
	}

	if squad.Status != models.SquadStatusPending {
		b.ChannelStatePublisher.PublishChannelOutOfLiveSquad(ctx, deletedMemberID)
	}
}
