package backend

import (
	"context"

	"code.justin.tv/twitch-events/meepo/internal/models"
)

// DeleteUser removes the channel from its current squad and deletes pending and rejected incoming invitations.
func (b *backend) DeleteUser(ctx context.Context, channelID string) error {
	// Note: Leaving current live squad and deleting invitations are disjoint operations and do not have to be executed
	// in a specific order.

	// Leave current live squad
	squad, err := b.Datastore.GetSquadByChannelID(ctx, channelID)
	if err != nil {
		return err
	}
	if squad != nil {
		if _, err = b.LeaveSquad(ctx, channelID, squad.ID, channelID, false); err != nil {
			return err
		}
	}

	// Delete pending and rejected invitations
	invitations, err := b.Datastore.DeleteInvitationsByRecipientID(
		ctx,
		channelID,
		[]models.InvitationStatus{models.InvitationStatusPending, models.InvitationStatusRejected},
	)
	if err != nil {
		return err
	}

	squads, err := b.getManagedSquadsByIDs(ctx, models.NewSquadIDsFromDBInvitations(invitations))
	if err != nil {
		b.log.LogCtx(ctx, "channel_id", channelID, "err", err, "error getting managed squads for pubsub for deleting user")
	} else {
		// Publish the updated squads to PubSub. (These pubsub messages will allow the squad owners'
		// widgets to stop rendering the deleted invitations.)
		b.publishSquadsToMembersAndSquad(ctx, squads)
	}

	return nil
}
