package backend

import (
	"context"
	"fmt"

	"code.justin.tv/feeds/errors"
	"github.com/twitchtv/twirp"

	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// LeaveSquad deletes a user's own membership from the squad, updates the owner of the squad if the user is the owner,
// and ends the squad if the user is the last member in the squad.
func (b *backend) LeaveSquad(ctx context.Context, memberID, squadID, callerID string, wentOffline bool) (*models.Squad, error) {
	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbSquad, err := b.Datastore.GetSquadByID(txCtx, squadID)
	if err != nil {
		return nil, err
	}
	if dbSquad == nil {
		if callerID != "" {
			b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
				ChannelID: callerID,
				SquadID:   squadID,
				Method:    models.ErrorMethodTypeUpdateSquad,
				ErrorCode: meepo_errors.ErrSquadNotFound,
			})
		}
		return nil, twirp.NotFoundError(fmt.Sprintf("Squad with id %v does not exist", squadID)).WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrSquadNotFound)
	}
	oldSquad := models.NewManagedSquadFromDB(dbSquad, nil, nil)

	// Delete member
	managedSquad, err := b.handleDeleteMemberDB(txCtx, memberID, dbSquad)
	if err != nil {
		return nil, err
	}

	// Delete invitations if owner was deleted
	deletedInvitations, err := b.handleDeletedSquadOwner(txCtx, memberID, dbSquad)
	if err != nil {
		return nil, err
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error deleting member")
	}

	b.deleteMemberFromCacheAndPublishUpdates(ctx, memberID, managedSquad)

	// Send a pubsub message to the recipients for deleted invitations.
	b.publishSquadInvitesToRecipientFromUpdatedInvitations(ctx, deletedInvitations)

	if managedSquad != nil {
		method := models.CreatorMethodTypeClickLeaveSquad
		if wentOffline {
			method = models.CreatorMethodTypeGoOffline
		}
		b.fireCreatorActionTrackingEvent(ctx, models.CreatorActionTrackingEventSet{
			SquadID:     managedSquad.ID,
			MemberIDs:   managedSquad.MemberIDs,
			OwnerID:     managedSquad.OwnerID,
			SquadStatus: &managedSquad.Status,
			Events: []models.CreatorActionTrackingEventInfo{
				{
					ChannelID:     memberID,
					CreatorAction: models.CreatorActionTypeLeaveSquad,
					CreatorMethod: method,
				},
			},
		})

		if managedSquad.OwnerID != nil && oldSquad.OwnerID != nil && managedSquad.OwnerID != oldSquad.OwnerID {
			// squad owner changed
			b.fireCreatorActionTrackingEvent(ctx, models.CreatorActionTrackingEventSet{
				SquadID:     managedSquad.ID,
				MemberIDs:   managedSquad.MemberIDs,
				OwnerID:     managedSquad.OwnerID,
				SquadStatus: &managedSquad.Status,
				Events: []models.CreatorActionTrackingEventInfo{
					{
						ChannelID:     *managedSquad.OwnerID,
						CreatorAction: models.CreatorActionTypeInheritOwnership,
						CreatorMethod: models.CreatorMethodTypeOriginalOwnerLeave,
					},
				},
			})
		}

		if managedSquad.Status == models.SquadStatusEnded {
			method := models.StateChangeMethodTypeLastMemberLeave
			if wentOffline {
				method = models.StateChangeMethodTypeLastMemberOffline
			}
			numMembers := 0
			b.fireSquadStateChangeTrackingEvent(ctx, models.SquadStateChangeTrackingEventInfo{
				SquadID:       managedSquad.ID,
				State:         models.SquadStatusEnded,
				PreviousState: &oldSquad.Status,
				Method:        method,
				LastUpdated:   &oldSquad.UpdatedAt,
				NumMembers:    &numMembers,
			})
		}
	}

	return models.NewSquadFromManagedSquad(managedSquad), nil
}
