package backend

import (
	"context"
	"strings"

	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/pkg/errors"
)

func (b *backend) isAlwaysOn(memberID string) bool {
	alwaysOnChannelsStr := b.config.alwaysOnChannels.Get()
	alwaysOnChannels := strings.Split(alwaysOnChannelsStr, ",")
	for _, channel := range alwaysOnChannels {
		if memberID == channel {
			return true
		}
	}
	return false
}

func (b *backend) filterForMembersOfOldLiveSquads(members []*models.DBMemberAndSquad) []*models.DBMemberAndSquad {
	gracePeriod := b.config.livecheckNewSquadGracePeriod.Get()
	filteredMembers := make([]*models.DBMemberAndSquad, 0, len(members))
	for _, member := range members {
		// Only thing that can change that will update UpdatedAt column in the Squad DB
		// table is Status. Since we are only concerned with Live squads, UpdatedAt
		// value should indicate when Status was changed from Pending to Live
		age := b.clock.Since(member.Squad.UpdatedAt)
		if age >= gracePeriod {
			filteredMembers = append(filteredMembers, member)
		}
	}

	return filteredMembers
}

// LivecheckChannels gets all channels that are currently in a squad, checks whether they are live, makes pending delete offline channels leav
// squad and marks offline channels as pending delete if they are in active state.
func (b *backend) LivecheckChannels(ctx context.Context) error {
	members, err := b.Datastore.GetMembersByStatuses(
		ctx,
		[]models.MemberStatus{models.MemberStatusActive, models.MemberStatusPendingDelete},
		[]models.SquadStatus{models.SquadStatusLive},
	)
	if err != nil {
		return err
	}

	members = b.filterForMembersOfOldLiveSquads(members)
	if len(members) == 0 {
		return nil
	}

	memberIDs := make([]string, 0, len(members))
	memberMap := make(map[string]*models.Member)
	for _, member := range members {
		if b.isAlwaysOn(member.Member.MemberID) {
			continue
		}

		if m, ok := memberMap[member.Member.MemberID]; ok {
			// we log when a membership with this channel already exists, which shouldn't happen
			b.log.LogCtx(ctx, "member_id", member.Member.MemberID, "first_squad_id", member.Member.SquadID, "second_squad_id", m.SquadID, "has multiple active memberships")
		} else {
			memberMap[member.Member.MemberID] = models.NewMemberFromDBMemberAndSquad(member)
			memberIDs = append(memberIDs, member.Member.MemberID)
		}
	}

	liveMemberIDs, err := b.Liveline.GetLiveChannelsByChannelIDs(ctx, memberIDs)
	if err != nil {
		return errors.Wrap(err, "could not query liveline for live channels")
	}

	// update all pending delete live channels to active
	for _, memberID := range liveMemberIDs {
		member := memberMap[memberID]
		if member.Status == models.MemberStatusPendingDelete {
			_, err = b.Datastore.UpdateMemberStatusByID(ctx, member.ID, models.MemberStatusActive)
			if err != nil {
				b.log.LogCtx(ctx, "member_id", member.MemberID, "squad_id", member.SquadID, "err", err, "failed to be marked as active")
			}
		}
	}

	// remove all live channels from the map
	for _, memberID := range liveMemberIDs {
		delete(memberMap, memberID)
	}

	leaveCount := int64(0)
	for _, member := range memberMap {
		if member.Status == models.MemberStatusActive {
			// we don't delete an active offline member immediately to give the member a buffer time to come back online,
			// where the buffer time is the interval between two consecutive calls to livecheck channels
			_, err = b.Datastore.UpdateMemberStatusByID(ctx, member.ID, models.MemberStatusPendingDelete)
			if err != nil {
				b.log.LogCtx(ctx, "member_id", member.MemberID, "squad_id", member.SquadID, "err", err, "failed to be marked as pending delete")
			}
		} else {
			_, err := b.LeaveSquad(ctx, member.MemberID, member.SquadID, "", true)
			if err != nil {
				b.log.LogCtx(ctx, "member_id", member.MemberID, "squad_id", member.SquadID, "err", err, "failed to be deleted")
			} else {
				leaveCount++
			}
		}
	}

	b.stats.IncC("livecheck_channels.leave_count", leaveCount, 1)

	return nil
}
