package backend

import (
	"context"

	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// prepareToCreateInvitation:
// 1. validates that the sender can extend an invitation
// 2. validates that the squad is not full
// 3. validates that the recipient is not already in a squad
// 4. ensures that recipient does not have too many incoming pending invitations
// If invitations are rejected, they will be returned
func (b *backend) prepareToCreateInvitation(txCtx context.Context, squad *models.DBSquad, senderID string, recipientID string) ([]*models.DBInvitation, error) {
	// Check if the sender is allowed to send an invitation.
	isAuthed := b.isOwnerWithSquad(senderID, squad)
	if !isAuthed {
		var squadID string
		if squad != nil {
			squadID = squad.ID
		}
		b.FireSquadStreamErrorTrackingEvent(txCtx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       senderID,
			TargetChannelID: &recipientID,
			SquadID:         squadID,
			Method:          models.ErrorMethodTypePrepareToCreateInvitation,
			ErrorCode:       meepo_errors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "You cannot modify the squad").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrUnauthorized)
	}

	// Check that the total number of invitations and members is not at the maximum.
	isFull, err := b.checkSquadAtCapacity(txCtx, squad)
	if err != nil {
		return nil, err
	}
	if isFull {
		var squadID string
		if squad != nil {
			squadID = squad.ID
		}
		b.FireSquadStreamErrorTrackingEvent(txCtx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       senderID,
			TargetChannelID: &recipientID,
			SquadID:         squadID,
			Method:          models.ErrorMethodTypePrepareToCreateInvitation,
			ErrorCode:       meepo_errors.ErrSquadFull,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The squad is already full").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrSquadFull)
	}

	// Check if the recipient is already in the squad.
	recipientSquad, err := b.Datastore.GetSquadByChannelID(txCtx, recipientID)
	if err != nil {
		return nil, err
	}
	if recipientSquad != nil && recipientSquad.ID == squad.ID {
		b.FireSquadStreamErrorTrackingEvent(txCtx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       senderID,
			TargetChannelID: &recipientID,
			SquadID:         squad.ID,
			Method:          models.ErrorMethodTypePrepareToCreateInvitation,
			ErrorCode:       meepo_errors.ErrUserAlreadyInSquad,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The recipient is already in the squad").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrUserAlreadyInSquad)
	}

	// Delete rejected invitation from the squad to the recipient, if one exists
	rejectedInvitationsForSquad, err := b.Datastore.GetInvitationsBySquadID(txCtx, squad.ID, []models.InvitationStatus{models.InvitationStatusRejected})
	if err != nil {
		return nil, err
	}
	for _, invitation := range rejectedInvitationsForSquad {
		if invitation.RecipientID == recipientID {
			_, err = b.Datastore.UpdateInvitationStatus(txCtx, invitation.ID, models.InvitationStatusDeleted)
			if err != nil {
				return nil, err
			}
		}
	}

	// Reject last updated invitation for recipient if exceeds max incoming pending invitations per user
	pendingInvitationsForRecipient, err := b.Datastore.GetInvitationsByRecipientID(txCtx, recipientID, []models.InvitationStatus{models.InvitationStatusPending})
	if err != nil {
		return nil, err
	}

	// Check if at or exceed max
	var rejectedInvitations []*models.DBInvitation
	if len(pendingInvitationsForRecipient) < MaxPendingInvitationsPerChannel {
		return nil, nil
	}
	for _, invitation := range pendingInvitationsForRecipient[MaxPendingInvitationsPerChannel-1:] {
		inv, err := b.Datastore.RejectInvitation(txCtx, invitation.ID, models.InvitationRecipientMaxLimitReached)
		if err != nil {
			return nil, err
		}
		rejectedInvitations = append(rejectedInvitations, inv)
	}
	return rejectedInvitations, nil
}
