package backend

import (
	"context"
	"fmt"
	"sync"

	"code.justin.tv/twitch-events/meepo/internal/models"
)

const (
	// ProfileImageSize is the size of a user's profile image we send to pubsub
	ProfileImageSize = "150x150"
)

func (b *backend) publishSquadToMembersAndSquad(ctx context.Context, squad *models.ManagedSquad) {
	if squad == nil {
		return
	}

	pubsubUserSet, err := b.loadPubsubUsers(ctx, squad)
	if err != nil {
		b.log.LogCtx(ctx, "squad_id", squad.ID, "err", err, "error retrieving user information")
		return
	}

	pubsubMembers := make([]models.PubsubUser, len(squad.MemberIDs))
	for i, memberID := range squad.MemberIDs {
		member, ok := pubsubUserSet[memberID]
		if !ok {
			b.log.LogCtx(ctx, "user_id", memberID, "error retrieving user information")
			return
		}
		pubsubMembers[i] = member
	}

	pubsubSqInvitations := make([]models.PubsubOutgoingInvitation, len(squad.Invitations))
	for i, invitation := range squad.Invitations {
		recipient, ok := pubsubUserSet[invitation.RecipientID]
		if !ok {
			b.log.LogCtx(ctx, "user_id", invitation.RecipientID, "error retrieving user information")
			return
		}

		sender, ok := pubsubUserSet[invitation.SenderID]
		if !ok {
			b.log.LogCtx(ctx, "user_id", invitation.SenderID, "error retrieving user information")
			return
		}

		pubsubSqInvitations[i] = *models.NewPubsubOutgoingInvitation(
			invitation, recipient, sender)
	}

	pubsubSquadForOwner := models.NewPubsubManagedSquad(squad, pubsubMembers, pubsubSqInvitations)

	err = b.Pubsub.PublishSquadUpdate(ctx, pubsubSquadForOwner)
	if err != nil {
		b.log.LogCtx(ctx, "squad_id", squad.ID, "err", err, "error sending squad pubsub message")
	}
}

func (b *backend) publishSquadsToMembersAndSquad(ctx context.Context, squads []*models.ManagedSquad) {
	if squads == nil {
		return
	}

	wg := sync.WaitGroup{}
	wg.Add(len(squads))
	for _, squad := range squads {
		go func(sq *models.ManagedSquad) {
			defer wg.Done()
			b.publishSquadToMembersAndSquad(ctx, sq)
		}(squad)
	}
	wg.Wait()
}

func (b *backend) loadPubsubUsers(ctx context.Context, squad *models.ManagedSquad) (map[string]models.PubsubUser, error) {
	if squad == nil {
		return map[string]models.PubsubUser{}, nil
	}

	memberIDs := squad.MemberIDs
	invitations := squad.Invitations

	userIDSet := make(map[string]bool, len(memberIDs)+len(invitations))
	for _, userID := range memberIDs {
		userIDSet[userID] = true
	}
	for _, invitation := range invitations {
		userIDSet[invitation.RecipientID] = true
	}

	userIDs := make([]string, 0, len(userIDSet))
	for userID := range userIDSet {
		userIDs = append(userIDs, userID)
	}
	if len(userIDs) == 0 {
		return map[string]models.PubsubUser{}, nil
	}

	users, err := b.Users.GetUsersByIDs(ctx, userIDs)
	if err != nil {
		return nil, err
	}

	pubsubUserSet := make(map[string]models.PubsubUser, len(users))
	for _, user := range users {
		var imageURL *string
		if user.ProfileImageURL != nil {
			profileImageURLWithSize := fmt.Sprintf(*user.ProfileImageURL, ProfileImageSize)
			imageURL = &profileImageURLWithSize
		}
		pubsubUserSet[user.ID] = models.PubsubUser{
			ID:                 user.ID,
			DisplayName:        user.Displayname,
			Login:              user.Login,
			ProfileImageURL150: imageURL,
		}
	}

	return pubsubUserSet, nil
}

func (b *backend) publishSquadInvitesToRecipient(ctx context.Context, recipientID string) {
	if recipientID == "" {
		return
	}

	// get all pending invitations for the recipient
	invitations, err := b.GetPendingInvitationsByRecipientID(ctx, recipientID)
	if err != nil {
		b.log.LogCtx(ctx, "recipient_id", recipientID, "err", err, "error retrieving user invitations for pubsub")
		return
	}

	pubsubInvitations := make([]*models.PubsubIncomingInvitation, len(invitations))
	ids := make([]string, 0, len(invitations)+1)
	for _, invitation := range invitations {
		ids = append(ids, invitation.SenderID)
	}

	if len(ids) > 0 {
		ids = append(ids, recipientID)
		// hydrate invitations with necessary information
		users, getErr := b.Users.GetUsersByIDs(ctx, ids)
		if getErr != nil {
			b.log.LogCtx(ctx, "ids", ids, "err", getErr, "error retrieving user information")
			return
		}

		userMap := make(map[string]models.PubsubUser)
		for _, user := range users {
			pubsubUser := models.PubsubUser{
				ID:          user.ID,
				DisplayName: user.Displayname,
				Login:       user.Login,
			}
			if user.ProfileImageURL != nil {
				profileImageURLWithSize := fmt.Sprintf(*user.ProfileImageURL, ProfileImageSize)
				pubsubUser.ProfileImageURL150 = &profileImageURLWithSize
			}

			userMap[user.ID] = pubsubUser
		}

		for i, invitation := range invitations {
			sender, ok := userMap[invitation.SenderID]
			if !ok {
				b.log.LogCtx(ctx, "id", invitation.SenderID, "error retrieving user information")
				return
			}
			recipient, ok := userMap[invitation.RecipientID]
			if !ok {
				b.log.LogCtx(ctx, "id", invitation.RecipientID, "error retrieving user information")
				return
			}

			pubsubInvitations[i] = &models.PubsubIncomingInvitation{
				ID:          invitation.ID,
				SquadID:     invitation.SquadID,
				Sender:      sender,
				Recipient:   recipient,
				Status:      invitation.Status,
				NetworkType: invitation.NetworkType,

				CreatedAt: invitation.CreatedAt,
				UpdatedAt: invitation.UpdatedAt,
			}

		}
	}

	err = b.Pubsub.PublishIncomingSquadInvites(ctx, recipientID, pubsubInvitations)
	if err != nil {
		b.log.LogCtx(ctx, "recipient_id", recipientID, "err", err, "error sending channel squad invites pubsub message")
	}
}

// After updating an invitation from pending to anything else,
// send pubsub message to each recipient of the updated invitations
func (b *backend) publishSquadInvitesToRecipientFromUpdatedInvitations(ctx context.Context, updatedInvitations []*models.DBInvitation) {
	if len(updatedInvitations) == 0 {
		return
	}

	wg := sync.WaitGroup{}
	wg.Add(len(updatedInvitations))
	for _, updatedInvitation := range updatedInvitations {
		go func(inv *models.DBInvitation) {
			defer wg.Done()
			b.publishSquadInvitesToRecipient(ctx, inv.RecipientID)
		}(updatedInvitation)
	}
	wg.Wait()
}
