package backend

import (
	"context"

	"code.justin.tv/feeds/errors"
	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// RejectInvitation updates the status of an invitation from PENDING to REJECTED.
// It also cleans up already rejected invitations, by updating their status to DELETED
// so that each squad does not have more than 3 rejected invitation at any given time
func (b *backend) RejectInvitation(ctx context.Context, invitationID, callerID string, reasonRejected models.InvitationReasonRejected) (*models.Invitation, error) {
	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	invitation, err := b.rejectInvitationInDB(txCtx, invitationID, callerID, reasonRejected)
	if err != nil {
		return nil, err
	}

	squad, err := b.getManagedSquadByID(txCtx, invitation.SquadID, nil)
	if err != nil {
		return nil, err
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error rejecting invitation")
	}

	// Publish the updated squad to PubSub.  (This pubsub message message will allow the squad owner's
	// widget to render the rejected invitation.)
	b.publishSquadToMembersAndSquad(ctx, squad)

	// Publish updated invitation status to recipient
	b.publishSquadInvitesToRecipient(ctx, invitation.RecipientID)

	if squad != nil {
		b.fireCreatorActionTrackingEvent(ctx, models.CreatorActionTrackingEventSet{
			SquadID:      squad.ID,
			InvitationID: invitationID,
			MemberIDs:    squad.MemberIDs,
			OwnerID:      squad.OwnerID,
			SquadStatus:  &squad.Status,
			Events: []models.CreatorActionTrackingEventInfo{
				{
					ChannelID:     callerID,
					CreatorAction: models.CreatorActionTypeRejectInvite,
				},
			},
		})
	}

	// Return newly rejected invitation
	return models.NewInvitationFromDB(invitation), nil
}

// rejectInvitationInDB performs the database operations related to rejecting an invitation, and returns
// the updated invitation, and managed squad related to the invitation.
func (b *backend) rejectInvitationInDB(ctx context.Context, invitationID, callerID string, reasonRejected models.InvitationReasonRejected) (*models.DBInvitation, error) {
	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbInvitation, err := b.Datastore.GetInvitationByID(txCtx, invitationID)
	if err != nil {
		return nil, err
	}
	if dbInvitation == nil {
		b.FireSquadStreamErrorTrackingEvent(txCtx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			InviteID:  invitationID,
			Method:    models.ErrorMethodTypeRejectInvitation,
			ErrorCode: meepo_errors.ErrInvitationNotFound,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The invitation does not exist").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationNotFound)
	}

	// Check invitation status, can only reject PENDING invitation
	invitationStatus := models.NewInvitationStatusFromDB(dbInvitation.Status)
	if invitationStatus != models.InvitationStatusPending {
		b.FireSquadStreamErrorTrackingEvent(txCtx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &dbInvitation.RecipientID,
			SquadID:         dbInvitation.SquadID,
			InviteID:        invitationID,
			Method:          models.ErrorMethodTypeRejectInvitation,
			ErrorCode:       meepo_errors.ErrInvitationCannotBeRejected,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The invitation is not in rejected nor pending state").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationCannotBeRejected)
	}

	// Reject invitation with reason from param
	newDBInvitation, err := b.Datastore.RejectInvitation(txCtx, invitationID, reasonRejected)
	if err != nil {
		return nil, err
	}
	if newDBInvitation == nil {
		b.FireSquadStreamErrorTrackingEvent(txCtx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &dbInvitation.RecipientID,
			InviteID:        invitationID,
			Method:          models.ErrorMethodTypeRejectInvitation,
			ErrorCode:       meepo_errors.ErrInvitationNotFound,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "The invitation does not exist").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrInvitationNotFound)
	}

	// Cleanup rejected invitations, update status to deleted if there's too many rejected invitations
	_, err = b.deleteRejectedInvitationsBySquadID(txCtx, dbInvitation.SquadID, MaxRejectedInvitationsPerSquad)
	if err != nil {
		return nil, err
	}

	// Return newly rejected invitation
	return newDBInvitation, nil
}
