package backend

import (
	"context"

	service_common "code.justin.tv/feeds/service-common"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// RejectOutOfNetworkInvitations updates the status of a channel's out-of-network invitations from PENDING to REJECTED.
// It also cleans up already rejected invitations, by updating their status to DELETED
// so that each squad does not have more than 3 rejected invitations at any given time.
func (b *backend) RejectOutOfNetworkInvitations(ctx context.Context, channelID, callerID string, reasonRejected models.InvitationReasonRejected) error {
	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbInvitations, err := b.Datastore.GetInvitationsByRecipientID(
		txCtx,
		channelID,
		[]models.InvitationStatus{models.InvitationStatusPending},
	)

	if err != nil {
		return err
	}

	isInNetworks, networkErrs := b.areInNetworkInvitations(txCtx, dbInvitations)

	outOfNetworkInvitations := make([]*models.DBInvitation, 0, len(dbInvitations))
	for idx, dbInvitation := range dbInvitations {
		isInNetwork, networkErr := isInNetworks[idx], networkErrs[idx]
		if networkErr != nil {
			b.log.LogCtx(txCtx, "recipient_id", dbInvitation.RecipientID, "sender_id", dbInvitation.SenderID, "err", networkErr, "error checking network information")
			continue
		}
		if !isInNetwork {
			outOfNetworkInvitations = append(outOfNetworkInvitations, dbInvitation)
		}
	}

	rejectedInvitations := make([]*models.DBInvitation, 0, len(outOfNetworkInvitations))
	rejectErrs := make([]error, 0, len(outOfNetworkInvitations))
	for idx, dbInvitation := range outOfNetworkInvitations {
		rejectedInv, rejectErr := b.rejectInvitationInDB(txCtx, dbInvitation.ID, callerID, reasonRejected)
		if rejectErr != nil {
			rejectErrs[idx] = rejectErr
			continue
		}

		rejectedInvitations = append(rejectedInvitations, rejectedInv)
	}

	rejectSquads, err := b.getManagedSquadsByIDs(txCtx, models.NewSquadIDsFromDBInvitations(rejectedInvitations))
	if err != nil {
		return err
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return errors.Wrap(err, "error rejecting out of network invitations")
	}

	// Publish the updated squads to PubSub.  (This pubsub message messages will allow the squad owners'
	// widgets to render the rejected invitations.)
	b.publishSquadsToMembersAndSquad(ctx, rejectSquads)
	// Send a pubsub message to the recipients for deleted invitations.
	b.publishSquadInvitesToRecipientFromUpdatedInvitations(ctx, rejectedInvitations)

	// Track creator action rejected invitations
	eventSets := make([]models.CreatorActionTrackingEventSet, 0, len(rejectSquads))
	for i, squad := range rejectSquads {
		eventSets = append(eventSets, models.CreatorActionTrackingEventSet{
			SquadID:      squad.ID,
			InvitationID: rejectedInvitations[i].ID,
			MemberIDs:    squad.MemberIDs,
			OwnerID:      squad.OwnerID,
			SquadStatus:  &squad.Status,
			Events: []models.CreatorActionTrackingEventInfo{
				{
					ChannelID:     callerID,
					CreatorAction: models.CreatorActionTypeRejectInvite,
				},
			},
		})
	}
	b.fireCreatorActionTrackingEvents(ctx, eventSets)

	// Return errors, if any
	return service_common.ConsolidateErrors(rejectErrs)
}
