package backend

import (
	"context"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// RemoveMember deletes a user's membership from the squad as another user.
func (b *backend) RemoveMember(ctx context.Context, memberID, squadID, callerID string) (*models.Squad, error) {
	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbSquad, err := b.Datastore.GetSquadByID(txCtx, squadID)
	if err != nil {
		return nil, err
	}

	squad, err := b.handleDeleteMemberDB(txCtx, memberID, dbSquad)
	if err != nil {
		return nil, err
	}

	// Delete invitations if owner was deleted
	deletedInvitations, err := b.handleDeletedSquadOwner(txCtx, memberID, dbSquad)
	if err != nil {
		return nil, err
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error deleting member")
	}

	b.deleteMemberFromCacheAndPublishUpdates(ctx, memberID, squad)

	// Send a pubsub message to the recipients for deleted invitations.
	b.publishSquadInvitesToRecipientFromUpdatedInvitations(ctx, deletedInvitations)

	if squad != nil {
		b.fireCreatorActionTrackingEvent(ctx, models.CreatorActionTrackingEventSet{
			SquadID:     squad.ID,
			MemberIDs:   squad.MemberIDs,
			OwnerID:     squad.OwnerID,
			SquadStatus: &squad.Status,
			Events: []models.CreatorActionTrackingEventInfo{
				{
					ChannelID:     memberID,
					CreatorAction: models.CreatorActionTypeRemoveFromSquad,
					CreatorMethod: models.CreatorMethodTypeOwnerClickRemove,
				},
			},
		})
	}

	return models.NewSquadFromManagedSquad(squad), nil
}
