package backend

import (
	"context"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/twitch-events/meepo/internal/models"
	usersmodels "code.justin.tv/web/users-service/models"
	"github.com/pkg/errors"
)

func (b *backend) fireCreatorActionTrackingEvent(ctx context.Context, eventSet models.CreatorActionTrackingEventSet) {
	// Get squad information if we need it.
	var squad *models.Squad
	var err error
	if eventSet.OwnerID == nil || eventSet.MemberIDs == nil || eventSet.SquadStatus == nil {
		squad, err = b.GetSquadByID(ctx, eventSet.SquadID)
		if err != nil {
			b.log.LogCtx(ctx, "error firing creator_action_tracking_event=", errors.Wrap(err, "failed to fetch squad by id"))
			return
		}
	}

	// for each event, if override exists then use override. else use fetched value.
	userSlice := make([]string, 0, len(eventSet.Events))
	livenessSlice := make([]string, 0, len(eventSet.Events))
	for _, event := range eventSet.Events {
		// Create a list of the users for whom we need user info.
		if event.DisplayName == nil {
			userSlice = append(userSlice, event.ChannelID)
		}
		// Create a list of the users for whom we need liveness info.
		if event.ChannelIsLive == nil {
			livenessSlice = append(livenessSlice, event.ChannelID)
		}
	}

	// Create a map with the displayname for a user mapped to their ID.
	channelIDToDisplayNameMap := make(map[string]string)
	if len(userSlice) > 0 {
		users, err := b.Clients.Users.GetUsersByIDs(ctx, userSlice)
		if err != nil {
			b.log.LogCtx(ctx, "error firing creator_action_tracking_event=", errors.Wrap(err, "failed to get user displaynames"))
			return
		}

		for _, user := range users {
			if user.Displayname != nil {
				channelIDToDisplayNameMap[user.ID] = *(user.Displayname)
			}
		}
	}

	// Create a map with all the liveness info for a user mapped to their ID.
	channelIsLiveMap := make(map[string]bool)
	if len(livenessSlice) > 0 {
		liveUsers, err := b.Clients.Liveline.GetLiveChannelsByChannelIDs(ctx, livenessSlice)
		if err != nil {
			b.log.LogCtx(ctx, "error firing creator_action_tracking_event=", errors.Wrap(err, "failed to get live channels"))
			return
		}

		for _, id := range liveUsers {
			channelIsLiveMap[id] = true
		}
	}

	// Loop over all events and fire them off.
	for _, event := range eventSet.Events {
		displayName := ""
		if event.DisplayName == nil {
			displayName = channelIDToDisplayNameMap[event.ChannelID]
		} else {
			displayName = *(event.DisplayName)
		}

		var numMembers int
		if eventSet.MemberIDs == nil && squad != nil {
			numMembers = len(squad.MemberIDs)
		} else {
			numMembers = len(eventSet.MemberIDs)
		}

		var isLive bool
		if event.ChannelIsLive == nil {
			isLive = channelIsLiveMap[event.ChannelID]
		} else {
			isLive = *(event.ChannelIsLive)
		}

		isOwner := false
		if eventSet.OwnerID == nil && squad != nil && squad.OwnerID != nil {
			isOwner = *(squad.OwnerID) == event.ChannelID
		} else if eventSet.OwnerID != nil {
			isOwner = *(eventSet.OwnerID) == event.ChannelID
		}

		var squadViewable bool
		if eventSet.SquadStatus == nil && squad != nil {
			squadViewable = squad.Status == models.SquadStatusLive
		} else {
			squadViewable = *(eventSet.SquadStatus) == models.SquadStatusLive
		}

		channelID, err := strconv.Atoi(event.ChannelID)
		if err != nil {
			b.log.LogCtx(ctx, "error firing creator_action_tracking_event=", errors.Wrap(err, "failed to convert channelID from str to int"))
			return
		}

		err = b.Clients.Spade.TrackEvent(ctx, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
			ChannelID:     channelID,
			Channel:       displayName,
			NumMembers:    numMembers,
			SquadStreamID: eventSet.SquadID,
			IsLive:        isLive,
			IsOwner:       isOwner,
			SquadViewable: squadViewable,
			InviteID:      eventSet.InvitationID,
			Action:        event.CreatorAction,
			Method:        event.CreatorMethod,
		})
		if err != nil {
			b.log.LogCtx(ctx, "error firing creator_action_tracking_event=", errors.Wrap(err, "failed to send tracking event"))
			return
		}
	}
}

func (b *backend) fireCreatorActionTrackingEvents(ctx context.Context, eventSets []models.CreatorActionTrackingEventSet) {
	if len(eventSets) == 0 {
		return
	}

	wg := sync.WaitGroup{}
	wg.Add(len(eventSets))
	for _, eventSet := range eventSets {
		go func(param models.CreatorActionTrackingEventSet) {
			defer wg.Done()
			b.fireCreatorActionTrackingEvent(ctx, param)
		}(eventSet)
	}
	wg.Wait()
}

func (b *backend) fireSquadStateChangeTrackingEvent(ctx context.Context, event models.SquadStateChangeTrackingEventInfo) {
	var squad *models.Squad
	var err error
	squad, err = b.GetSquadByID(ctx, event.SquadID)
	if err != nil {
		b.log.LogCtx(ctx, "error firing squad_state_change_tracking_event=", errors.Wrap(err, "failed to fetch squad by id"))
		return
	}

	var owner *usersmodels.Properties
	if event.State != models.SquadStatusEnded && squad != nil && squad.OwnerID != nil {
		users, err := b.Clients.Users.GetUsersByIDs(ctx, []string{*(squad.OwnerID)})
		if err != nil {
			b.log.LogCtx(ctx, "error firing squad_state_change_tracking_event=", errors.Wrap(err, "failed to send tracking event"))
			return
		}
		if len(users) > 0 && users[0] != nil {
			owner = users[0]
		}
	}

	numMembersLive := 0
	if len(squad.MemberIDs) > 0 {
		liveUsers, err := b.Clients.Liveline.GetLiveChannelsByChannelIDs(ctx, squad.MemberIDs)
		if err != nil {
			b.log.LogCtx(ctx, "error firing squad_state_change_tracking_event=", errors.Wrap(err, "failed to determine live channels"))
			return
		}
		numMembersLive = len(liveUsers)
	}

	var intOwnerChannelID int
	if owner != nil {
		intOwnerChannelID, err = strconv.Atoi(owner.ID)
		if err != nil {
			b.log.LogCtx(ctx, "error firing squad_state_change_tracking_event=", errors.Wrap(err, "failed to convert channel id from string to int"))
			return
		}
	}

	var displayName string
	if owner != nil && owner.Displayname != nil {
		displayName = *(owner.Displayname)
	}

	var previousState models.SquadStatus
	if event.PreviousState != nil {
		previousState = *event.PreviousState
	}

	var lastUpdated *time.Time
	if event.LastUpdated != nil {
		tempTime := *(event.LastUpdated)
		tempTime = tempTime.UTC()
		lastUpdated = &tempTime
	}

	var numMembers int
	if event.NumMembers != nil {
		numMembers = *(event.NumMembers)
	} else {
		numMembers = len(squad.MemberIDs)
	}

	err = b.Clients.Spade.TrackEvent(ctx, models.SquadStateChangeTrackingEvent, models.SquadStateChangeTracking{
		SquadStreamID:  event.SquadID,
		State:          event.State,
		PreviousState:  previousState,
		OwnerChannelID: intOwnerChannelID,
		OwnerChannel:   displayName,
		LastUpdated:    lastUpdated,
		NumMembers:     numMembers,
		NumMembersLive: numMembersLive,
		Method:         event.Method,
	})
	if err != nil {
		b.log.LogCtx(ctx, "error firing squad_state_change_tracking_event=", errors.Wrap(err, "failed to send tracking event"))
		return
	}
}

func (b *backend) FireSquadStreamErrorTrackingEvent(ctx context.Context, event models.SquadStreamErrorTrackingEventInfo) {
	var squad *models.Squad
	var err error

	if event.NumMembers == nil {
		squad, err = b.GetSquadByID(ctx, event.SquadID)
		if err != nil {
			b.log.LogCtx(ctx, "error firing squad_stream_error_tracking_event=", errors.Wrap(err, "failed to fetch squad"))
			return
		}
	}

	channelIDs := []string{event.ChannelID}
	if event.TargetChannelID != nil {
		channelIDs = append(channelIDs, *event.TargetChannelID)
	}
	users, err := b.Clients.Users.GetUsersByIDs(ctx, channelIDs)
	if err != nil {
		b.log.LogCtx(ctx, "error firing squad_stream_error_tracking_event=", errors.Wrap(err, "failed to get user by channelID"))
		return
	}

	intChannelID, err := strconv.Atoi(event.ChannelID)
	if err != nil {
		b.log.LogCtx(ctx, "error firing squad_stream_error_tracking_event=", errors.Wrap(err, "failed to convert channelID from string to int"))
		return
	}

	var intTargetChannelID int
	if event.TargetChannelID != nil {
		intTargetChannelID, err = strconv.Atoi(*event.TargetChannelID)
		if err != nil {
			b.log.LogCtx(ctx, "error firing squad_stream_error_tracking_event=", errors.Wrap(err, "failed to convert targetChannelID from string to int"))
			return
		}
	}

	var displayName string
	var targetDisplayName string
	if event.TargetChannelID != nil && *event.TargetChannelID == event.ChannelID {
		if len(users) > 0 && users[0] != nil && users[0].Displayname != nil {
			displayName = *users[0].Displayname
			targetDisplayName = displayName
		}
	} else {
		for _, user := range users {
			if user != nil && user.Displayname != nil {
				if user.ID == event.ChannelID {
					displayName = *user.Displayname
				}
				if event.TargetChannelID != nil && user.ID == *event.TargetChannelID {
					targetDisplayName = *user.Displayname
				}
			}
		}
	}

	numMembers := -1
	if event.NumMembers != nil {
		numMembers = *event.NumMembers
	} else if squad != nil {
		numMembers = len(squad.MemberIDs)
	}

	err = b.Clients.Spade.TrackEvent(ctx, models.SquadStreamErrorTrackingEvent, models.SquadStreamErrorTracking{
		ChannelID:       intChannelID,
		Channel:         displayName,
		TargetChannelID: intTargetChannelID,
		TargetChannel:   targetDisplayName,
		NumMembers:      numMembers,
		SquadStreamID:   event.SquadID,
		InviteID:        event.InviteID,
		ErrorCode:       event.ErrorCode,
		Method:          event.Method,
	})
	if err != nil {
		b.log.LogCtx(ctx, "error firing squad_stream_error_tracking_event=", errors.Wrap(err, "failed to send tracking event"))
		return
	}
}
