package backend

import (
	"context"

	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// UpdateInvitePolicyByChannelID updates the invite policy for the given channel.
func (b *backend) UpdateInvitePolicyByChannelID(ctx context.Context, channelID, callerID string, invitePolicy models.InvitationPolicy) (*models.InvitePolicy, error) {
	if (callerID != channelID) && !b.isAdmin(callerID) {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &channelID,
			Method:          models.ErrorMethodTypeUpdateInvitePolicyByChannelID,
			ErrorCode:       meepo_errors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot see the channel's invite policy").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrUnauthorized)
	}

	return b.updateInvitePolicyByChannelIDAuthed(ctx, channelID, callerID, invitePolicy)
}

func (b *backend) updateInvitePolicyByChannelIDAuthed(ctx context.Context, channelID, callerID string, invitePolicy models.InvitationPolicy) (*models.InvitePolicy, error) {
	updatedDBInvitePolicy, err := b.Datastore.UpdateInvitePolicyByChannelID(ctx, channelID, callerID, invitePolicy)
	if err != nil {
		return nil, err
	} else if updatedDBInvitePolicy == nil {
		return nil, nil
	}

	updatedInvitePolicy := models.NewInvitePolicyFromDB(updatedDBInvitePolicy)
	return updatedInvitePolicy, nil
}
