package backend

import (
	"context"
	"fmt"

	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/pkg/errors"
	"github.com/twitchtv/twirp"
)

// UpdateSquad updates a squad's information.
func (b *backend) UpdateSquad(ctx context.Context, squadID string, status models.SquadStatus, callerID string) (*models.Squad, error) {
	if status != models.SquadStatusLive {
		// We currently only support setting a squad from PENDING to LIVE
		return nil, twirp.InvalidArgumentError("status", "must be LIVE").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrArgumentInvalid)
	}

	txCtx, createdTx, err := b.Datastore.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer b.Datastore.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbSquad, err := b.Datastore.GetSquadByID(ctx, squadID)
	if err != nil {
		return nil, err
	}
	if dbSquad == nil {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			SquadID:   squadID,
			Method:    models.ErrorMethodTypeUpdateSquad,
			ErrorCode: meepo_errors.ErrSquadNotFound,
		})
		return nil, twirp.NotFoundError(fmt.Sprintf("Squad with id %v does not exist", squadID)).WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrSquadNotFound)
	}
	lastUpdated := dbSquad.UpdatedAt

	if models.NewSquadStatusFromDB(dbSquad.Status) != models.SquadStatusPending {
		// We currently only support setting a squad from PENDING to LIVE
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			SquadID:   squadID,
			Method:    models.ErrorMethodTypeUpdateSquad,
			ErrorCode: meepo_errors.ErrSquadCannotBeUpdated,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "Squad must be in PENDING status").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrSquadCannotBeUpdated)
	}

	members, err := b.Datastore.GetMembersBySquadID(txCtx, squadID)
	if err != nil {
		return nil, err
	}

	if len(members) < 2 {
		b.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			SquadID:   squadID,
			Method:    models.ErrorMethodTypeUpdateSquad,
			ErrorCode: meepo_errors.ErrSquadTooFewMembers,
		})
		return nil, twirp.NewError(twirp.InvalidArgument, "Squad must have at least two members").WithMeta(meepo_errors.ErrMetaKey, meepo_errors.ErrSquadTooFewMembers)
	}

	updatedDBSquad, err := b.Datastore.UpdateSquadStatus(txCtx, squadID, status)
	if err != nil {
		return nil, err
	}

	if updatedDBSquad == nil {
		return nil, nil
	}

	managedSquad, err := b.getManagedSquadByID(txCtx, squadID, &preloadedSquadData{
		squad:   updatedDBSquad,
		members: members,
	})
	if err != nil {
		return nil, err
	}

	err = b.Datastore.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving squad")
	}

	squad := models.NewSquadFromManagedSquad(managedSquad)
	b.cacheSquad(ctx, squad)

	b.publishSquadToMembersAndSquad(ctx, managedSquad)

	// Send messages to SNS on the channels that are now part of a live squad.
	memberUserIDs := make([]string, len(members))
	for i, member := range members {
		memberUserIDs[i] = member.MemberID
	}
	b.ChannelStatePublisher.PublishChannelsInLiveSquad(ctx, managedSquad.ID, memberUserIDs)

	if squad != nil && status == models.SquadStatusLive {
		previousState := models.SquadStatusPending
		b.fireSquadStateChangeTrackingEvent(ctx, models.SquadStateChangeTrackingEventInfo{
			SquadID:       squad.ID,
			State:         models.SquadStatusLive,
			PreviousState: &previousState,
			Method:        models.StateChangeMethodTypeOwnerStartSquad,
			LastUpdated:   &lastUpdated,
		})

		if len(squad.MemberIDs) > 1 {
			events := make([]models.CreatorActionTrackingEventInfo, 0, len(squad.MemberIDs))
			for _, memberID := range squad.MemberIDs {
				creatorActionType := models.CreatorActionTypeJoinSquad
				if memberID == callerID {
					creatorActionType = models.CreatorActionTypeStartSquad
				}
				events = append(events, models.CreatorActionTrackingEventInfo{
					ChannelID:     memberID,
					CreatorAction: creatorActionType,
					CreatorMethod: models.CreatorMethodTypeOwnerClickStart,
				})
			}

			b.fireCreatorActionTrackingEvent(ctx, models.CreatorActionTrackingEventSet{
				SquadID:     squad.ID,
				MemberIDs:   squad.MemberIDs,
				OwnerID:     squad.OwnerID,
				SquadStatus: &squad.Status,
				Events:      events,
			})
		}
	}

	return squad, nil
}
