package cache

import (
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/gomemcache/memcache"
)

// Config contains the configuration values for the cache.
type Config struct {
	ConfigServer     *distconf.Str
	PollInterval     *distconf.Duration
	ReadWriteTimeout *distconf.Duration
	MaxIdleConns     *distconf.Int
}

// Load loads the config for the cache.
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.ConfigServer = dconf.Str("meepo.cache.config_server", "")
	if c.ConfigServer.Get() == "" {
		return errors.New("unable to find cache config server")
	}
	c.PollInterval = dconf.Duration("meepo.cache.poll_interval", time.Minute)
	c.ReadWriteTimeout = dconf.Duration("meepo.cache.rw_timeout", 350*time.Millisecond)
	c.MaxIdleConns = dconf.Int("meepo.cache.idle_conn_count", 1000)
	return nil
}

// Service runs the cache.
type Service struct {
	*memcache.Client
	config       *Config
	onClose      chan struct{}
	stopPollFunc func()
}

// NewCacheService creates a cache service based on the given config.
func NewCacheService(config *Config) (*Service, error) {
	return &Service{
		config: config,
	}, nil
}

// Setup sets up the cache based on the config.
func (s *Service) Setup() error {
	s.Client, s.stopPollFunc = memcache.Elasticache(s.config.ConfigServer.Get(), s.config.PollInterval.Get())
	s.Client.Timeout = s.config.ReadWriteTimeout.Get()
	s.Client.MaxIdleConns(int(s.config.MaxIdleConns.Get()))
	s.onClose = make(chan struct{})
	return nil
}

// Start starts the cache service.
func (s *Service) Start() error {
	// Wait for close signal
	<-s.onClose
	return nil
}

// Close ends the cache service.
func (s *Service) Close() error {
	s.stopPollFunc()
	close(s.onClose)
	return nil
}
