package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/util"
)

// createMember adds an membership for a user to a squad and returns the db representation of the membership.
func (d *datastore) createMember(ctx context.Context, input *models.CreateMemberInput) (*models.DBMember, error) {
	id, err := util.NewID()
	if err != nil {
		return nil, errors.Wrapf(err, "could not generate a new ID")
	}

	query := `
		INSERT INTO squad_members (
		  id, 
		  squad_id, 
		  member_id,
		  created_at,
		  status
		) VALUES (
			$1, $2, $3, $4, $5
		) RETURNING
		  id, squad_id, member_id, created_at, deleted_at, status`

	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		id,
		input.SquadID,
		input.MemberID,
		d.now(),
		models.NewDBFromMemberStatus(input.Status),
	)

	member := models.DBMember{}
	err = row.Scan(
		&member.ID,
		&member.SquadID,
		&member.MemberID,
		&member.CreatedAt,
		&member.DeletedAt,
		&member.Status,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not create membership for user ID %v in squad ID %v", input.MemberID, input.SquadID)
	}

	return &member, nil
}

// CreateMember records stats for createMember.
func (d *datastore) CreateMember(ctx context.Context, input *models.CreateMemberInput) (*models.DBMember, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "create_member", startTime, err == nil)
	}(time.Now())

	member, err := d.createMember(ctx, input)
	return member, err
}
