package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/util"
)

// createPendingInvitation adds a pending invitation for a user to a squad and returns the db representation of the invitation.
func (d *datastore) createPendingInvitation(ctx context.Context, input *models.CreatePendingInvitationInput) (*models.DBInvitation, error) {
	id, err := util.NewID()
	if err != nil {
		return nil, errors.Wrapf(err, "could not generate a new ID")
	}

	query := `
		INSERT INTO squad_invitations (
		  id, 
		  squad_id, 
		  sender_id,
		  recipient_id,
		  status,
		  reason_rejected,
		  created_at, 
		  updated_at
		) SELECT
			$1, $2::VARCHAR(64), $3, $4::VARCHAR(20), $5, $6, $7, $8
		WHERE NOT EXISTS (
		  SELECT * FROM squad_invitations WHERE squad_id = $2 AND recipient_id = $4 AND status = $5
		) RETURNING
		  id, squad_id, sender_id, recipient_id, status, reason_rejected, created_at, updated_at`

	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		id,
		input.SquadID,
		input.SenderID,
		input.RecipientID,
		models.NewDBFromInvitationStatus(models.InvitationStatusPending),
		nil,
		d.now(),
		d.now(),
	)

	invitation := models.DBInvitation{}
	err = row.Scan(
		&invitation.ID,
		&invitation.SquadID,
		&invitation.SenderID,
		&invitation.RecipientID,
		&invitation.Status,
		&invitation.ReasonRejected,
		&invitation.CreatedAt,
		&invitation.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not create invitation for user ID %v in squad ID %v", input.RecipientID, input.SquadID)
	}

	return &invitation, nil
}

// CreatePendingInvitation records stats for createPendingInvitation.
func (d *datastore) CreatePendingInvitation(ctx context.Context, input *models.CreatePendingInvitationInput) (*models.DBInvitation, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "create_pending_invitation", startTime, err == nil)
	}(time.Now())

	invitation, err := d.createPendingInvitation(ctx, input)
	return invitation, err
}
