package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/util"
)

// createSquad updates the squad in the db and returns the db representation of the updated squad.
func (d *datastore) createSquad(ctx context.Context, input *models.CreateSquadInput) (*models.DBSquad, error) {
	id, err := util.NewID()
	if err != nil {
		return nil, errors.Wrapf(err, "could not generate a new ID")
	}

	query := `
		INSERT INTO squads (
		  id, 
		  owner_id, 
		  status, 
		  created_by, 
		  created_at, 
		  updated_at
		) VALUES (
			$1, $2, $3, $4, $5, $6
		) RETURNING
		  id, owner_id, status, created_by, created_at, updated_at`

	creationTime := d.now()
	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		id,
		input.OwnerID,
		models.NewDBFromSquadStatus(input.Status),
		input.CreatedBy,
		creationTime,
		creationTime,
	)

	squad := models.DBSquad{}
	err = row.Scan(
		&squad.ID,
		&squad.OwnerID,
		&squad.Status,
		&squad.CreatedBy,
		&squad.CreatedAt,
		&squad.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not create squad for user ID %v", input.OwnerID)
	}

	return &squad, nil
}

// CreateSquad records stats for createSquad.
func (d *datastore) CreateSquad(ctx context.Context, input *models.CreateSquadInput) (*models.DBSquad, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "create_squad", startTime, err == nil)
	}(time.Now())

	squad, err := d.createSquad(ctx, input)
	return squad, err
}
