package datastore

import (
	"context"
	"database/sql"
	"fmt"
	"time"

	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/twitch-events/meepo/internal/clock"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// Datastore represents the interface for performing data operations.
type Datastore interface {
	GetInvitationsCountByRecipientID(ctx context.Context, channelID string, statuses []models.InvitationStatus) (int64, error)
	GetInvitationsByRecipientID(ctx context.Context, channelID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error)
	GetInvitationByID(ctx context.Context, invitationID string) (*models.DBInvitation, error)
	GetInvitationsBySquadID(ctx context.Context, squadID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error)
	GetInvitePolicyByChannelID(ctx context.Context, channelID string) (*models.DBInvitePolicy, error)
	GetMembersBySquadID(ctx context.Context, squadID string) ([]*models.DBMember, error)
	GetMembersByStatuses(ctx context.Context, statuses []models.MemberStatus, squadStatuses []models.SquadStatus) ([]*models.DBMemberAndSquad, error)
	GetSquadByChannelID(ctx context.Context, channelID string) (*models.DBSquad, error)
	GetSquadByID(ctx context.Context, id string) (*models.DBSquad, error)

	CreatePendingInvitation(ctx context.Context, input *models.CreatePendingInvitationInput) (*models.DBInvitation, error)
	CreateMember(ctx context.Context, input *models.CreateMemberInput) (*models.DBMember, error)
	CreateSquad(ctx context.Context, input *models.CreateSquadInput) (*models.DBSquad, error)
	DeleteInvitationsByRecipientID(ctx context.Context, channelID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error)
	DeleteInvitationsBySquadID(ctx context.Context, squadID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error)
	DeleteMember(ctx context.Context, memberID string) (*models.DBMember, error)
	EndSquad(ctx context.Context, squadID string) (*models.DBSquad, error)
	RejectInvitation(ctx context.Context, invitationID string, reasonRejected models.InvitationReasonRejected) (*models.DBInvitation, error)
	UpdateInvitationStatus(ctx context.Context, invitationID string, status models.InvitationStatus) (*models.DBInvitation, error)
	UpdateInvitePolicyByChannelID(ctx context.Context, channelID, callerID string, invitePolicy models.InvitationPolicy) (*models.DBInvitePolicy, error)
	UpdateMemberStatusByID(ctx context.Context, id string, status models.MemberStatus) (*models.DBMember, error)
	UpdateSquadOwner(ctx context.Context, squadID string, ownerID string) (*models.DBSquad, error)
	UpdateSquadStatus(ctx context.Context, squadID string, status models.SquadStatus) (*models.DBSquad, error)

	HardDeleteExpiredInvitations(ctx context.Context, deleteBefore time.Time) (int64, error)
	HardDeleteExpiredMembers(ctx context.Context, deleteBefore time.Time) (int64, error)
	HardDeleteExpiredSquads(ctx context.Context, deleteBefore time.Time) ([]*models.DBSquad, error)
	HardDeleteInvitePolicyByChannelID(ctx context.Context, channelID string) error

	StartOrJoinTx(ctx context.Context, opts *sql.TxOptions) (context.Context, bool, error)
	CommitTx(ctx context.Context, createdTx bool) error
	RollbackTxIfNotCommitted(ctx context.Context, createdTx bool)
}

// datastore contains the sql DB connection.
type datastore struct {
	*sql.DB

	log   *log.ElevatedLog
	stats *service_common.StatSender
	clock clock.Clock
}

// NewDatastore initializes a datastore to use with sql DB connection.
func NewDatastore(config *DBConfig, logger *log.ElevatedLog, stats *service_common.StatSender, clock clock.Clock) (Datastore, error) {
	logger.Log("hostname", config.hostname.Get(), "Opening connection to Postgres")

	connectionString := fmt.Sprintf("postgres://%s:%s@%s", config.username.Get(), config.password.Get(), config.hostname.Get())
	db, err := sql.Open(driver, connectionString)
	if err != nil {
		return nil, err
	}

	db.SetMaxOpenConns(int(config.maxOpenConns.Get()))

	return &datastore{
		DB:    db,
		log:   logger,
		stats: stats,
		clock: clock,
	}, nil
}
