package datastore

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// deleteInvitationsByRecipientID sets all pending and rejected invitations sent to the given channel as DELETED,
// and returns a list of deleted invitations.
func (d *datastore) deleteInvitationsByRecipientID(ctx context.Context, channelID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error) {
	query := fmt.Sprintf(`
		UPDATE squad_invitations
		SET 
		  status = $1,
		  reason_rejected = $2,
		  updated_at = $3
		WHERE recipient_id = $4
		AND status in (%s)
		RETURNING
		  id, squad_id, sender_id, recipient_id, status, reason_rejected, created_at, updated_at`,
		generatePlaceholders(5, len(statuses)))

	params := make([]interface{}, 4+len(statuses))
	params[0] = models.NewDBFromInvitationStatus(models.InvitationStatusDeleted)
	params[1] = models.NewDBFromInvitationReasonRejected(models.InvitationRecipientRejected)
	params[2] = d.now()
	params[3] = channelID
	for i, status := range statuses {
		params[i+4] = models.NewDBFromInvitationStatus(status)
	}

	rows, err := d.getTxIfJoined(ctx).QueryContext(ctx, query, params...)
	if err != nil {
		return nil, errors.Wrapf(err, "could not delete invitations for recipient ID %v", channelID)
	}
	defer d.closeRows(rows)

	var invitations []*models.DBInvitation
	for rows.Next() {
		invitation := models.DBInvitation{}
		err := rows.Scan(
			&invitation.ID,
			&invitation.SquadID,
			&invitation.SenderID,
			&invitation.RecipientID,
			&invitation.Status,
			&invitation.ReasonRejected,
			&invitation.CreatedAt,
			&invitation.UpdatedAt,
		)
		if err != nil {
			return nil, errors.Wrapf(err, "could not delete invitations for recipient ID %v", channelID)
		}
		invitations = append(invitations, &invitation)
	}

	return invitations, nil
}

// DeleteInvitationsByRecipientID records stats for deleteInvitationsByRecipientID.
func (d *datastore) DeleteInvitationsByRecipientID(ctx context.Context, channelID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "delete_invitations_by_recipient_id", startTime, err == nil)
	}(time.Now())

	invitations, err := d.deleteInvitationsByRecipientID(ctx, channelID, statuses)
	return invitations, err
}
