package datastore

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// deleteInvitationsBySquadID sets all invitations with the given statuses from the squad as DELETED and returns the
// invitations in the db representation.
func (d *datastore) deleteInvitationsBySquadID(ctx context.Context, squadID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error) {
	query := fmt.Sprintf(`
		UPDATE squad_invitations
		SET
   		  status = 'deleted'
		WHERE squad_id = $1
		AND status in (%s)
		RETURNING
		  id, squad_id, sender_id, recipient_id, status, reason_rejected, created_at, updated_at`,
		generatePlaceholders(2, len(statuses)))

	params := make([]interface{}, 1+len(statuses))
	params[0] = squadID
	for i, status := range statuses {
		params[i+1] = models.NewDBFromInvitationStatus(status)
	}

	rows, err := d.getTxIfJoined(ctx).QueryContext(ctx, query, params...)
	if err != nil {
		return nil, errors.Wrapf(err, "could not delete invitations for squad ID %v", squadID)
	}
	defer d.closeRows(rows)

	var invitations []*models.DBInvitation
	for rows.Next() {
		invitation := models.DBInvitation{}
		err := rows.Scan(
			&invitation.ID,
			&invitation.SquadID,
			&invitation.SenderID,
			&invitation.RecipientID,
			&invitation.Status,
			&invitation.ReasonRejected,
			&invitation.CreatedAt,
			&invitation.UpdatedAt,
		)
		if err != nil {
			return nil, errors.Wrapf(err, "could not delete invitations for squad ID %v", squadID)
		}
		invitations = append(invitations, &invitation)
	}

	return invitations, nil
}

// DeleteInvitationsBySquadID records stats for deleteInvitationsBySquadID.
func (d *datastore) DeleteInvitationsBySquadID(ctx context.Context, squadID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "delete_invitations_by_squad_id", startTime, err == nil)
	}(time.Now())

	invitations, err := d.deleteInvitationsBySquadID(ctx, squadID, statuses)
	return invitations, err
}
