package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// deleteMember deletes a membership for a user to their squad and returns the db representation of the deleted membership.
func (d *datastore) deleteMember(ctx context.Context, memberID string) (*models.DBMember, error) {
	query := `
		UPDATE squad_members
		SET 
		  deleted_at = $1,
		  status = 'deleted'
		WHERE member_id = $2
		AND status <> 'deleted'
		RETURNING
		  id, squad_id, member_id, created_at, deleted_at, status`

	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		d.now(),
		memberID,
	)

	member := models.DBMember{}
	err := row.Scan(
		&member.ID,
		&member.SquadID,
		&member.MemberID,
		&member.CreatedAt,
		&member.DeletedAt,
		&member.Status,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not delete membership for user ID %v", memberID)
	}

	return &member, nil
}

// DeleteMember records stats for deleteMember.
func (d *datastore) DeleteMember(ctx context.Context, memberID string) (*models.DBMember, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "delete_member", startTime, err == nil)
	}(time.Now())

	member, err := d.deleteMember(ctx, memberID)
	return member, err
}
