package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// endSquad ends the squad and returns the db representation of the ended squad.
func (d *datastore) endSquad(ctx context.Context, squadID string) (*models.DBSquad, error) {
	query := `
		UPDATE squads
		SET 
		  status = $1,
		  owner_id = NULL,
		  updated_at = $2
		WHERE id = $3
		RETURNING
		  id, owner_id, status, created_by, created_at, updated_at`

	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		models.NewDBFromSquadStatus(models.SquadStatusEnded),
		d.now(),
		squadID,
	)

	squad := models.DBSquad{}
	err := row.Scan(
		&squad.ID,
		&squad.OwnerID,
		&squad.Status,
		&squad.CreatedBy,
		&squad.CreatedAt,
		&squad.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not end squad for ID %v", squadID)
	}

	return &squad, nil
}

// EndSquad records stats for endSquad.
func (d *datastore) EndSquad(ctx context.Context, squadID string) (*models.DBSquad, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "end_squad", startTime, err == nil)
	}(time.Now())

	squad, err := d.endSquad(ctx, squadID)
	return squad, err
}
