package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// getInvitationByID returns db representation of the invitation with the provided ID.
func (d *datastore) getInvitationByID(ctx context.Context, invitationID string) (*models.DBInvitation, error) {
	query := `
		SELECT id, squad_id, sender_id, recipient_id, status, reason_rejected, created_at, updated_at
		FROM squad_invitations
		WHERE id = $1`

	row := d.getTxIfJoined(ctx).QueryRowContext(ctx, query, invitationID)
	invitation := models.DBInvitation{}
	err := row.Scan(
		&invitation.ID,
		&invitation.SquadID,
		&invitation.SenderID,
		&invitation.RecipientID,
		&invitation.Status,
		&invitation.ReasonRejected,
		&invitation.CreatedAt,
		&invitation.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not get invitation for ID %v", invitationID)
	}

	return &invitation, nil
}

// GetInvitationByID records stats for getInvitationByID.
func (d *datastore) GetInvitationByID(ctx context.Context, invitationID string) (*models.DBInvitation, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "get_invitation_by_id", startTime, err == nil)
	}(time.Now())

	invitation, err := d.getInvitationByID(ctx, invitationID)
	return invitation, err
}
