package datastore

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// getInvitationsByRecipientID returns a list of invitations to the given channel with the given statuses in the db representation.
func (d *datastore) getInvitationsByRecipientID(ctx context.Context, channelID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error) {
	if len(statuses) == 0 {
		return nil, errors.New("no statuses given")
	}

	query := fmt.Sprintf(`
		SELECT id, squad_id, sender_id, recipient_id, status, reason_rejected, created_at, updated_at
		FROM squad_invitations
		WHERE recipient_id = $1 
		AND status in (%s)
		ORDER BY updated_at DESC, created_at DESC, id`,
		generatePlaceholders(2, len(statuses)))

	params := make([]interface{}, 1+len(statuses))
	params[0] = channelID
	for i, status := range statuses {
		params[i+1] = models.NewDBFromInvitationStatus(status)
	}

	rows, err := d.getTxIfJoined(ctx).QueryContext(ctx, query, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get invitations")
	}
	defer d.closeRows(rows)

	var invitations []*models.DBInvitation
	for rows.Next() {
		invitation := models.DBInvitation{}
		err := rows.Scan(
			&invitation.ID,
			&invitation.SquadID,
			&invitation.SenderID,
			&invitation.RecipientID,
			&invitation.Status,
			&invitation.ReasonRejected,
			&invitation.CreatedAt,
			&invitation.UpdatedAt,
		)
		if err != nil {
			return nil, errors.Wrapf(err, "could not get invitation for channel ID %v and statuses %v", channelID, statuses)
		}
		invitations = append(invitations, &invitation)
	}

	return invitations, nil
}

// GetInvitationsByRecipientID records stats for getInvitationsByRecipientID.
func (d *datastore) GetInvitationsByRecipientID(ctx context.Context, channelID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "get_invitations_by_recipient_id", startTime, err == nil)
	}(time.Now())

	invitations, err := d.getInvitationsByRecipientID(ctx, channelID, statuses)
	return invitations, err
}
