package datastore

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// getInvitationsBySquadID returns a list of invitations from the given squad with the given statuses in the db representation.
func (d *datastore) getInvitationsBySquadID(ctx context.Context, squadID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error) {
	if len(statuses) == 0 {
		return nil, errors.New("no statuses given")
	}

	query := fmt.Sprintf(`
		SELECT id, squad_id, sender_id, recipient_id, status, reason_rejected, created_at, updated_at
		FROM squad_invitations
		WHERE squad_id = $1 
		AND status in (%s)
		ORDER BY updated_at ASC, created_at ASC`,
		generatePlaceholders(2, len(statuses)))

	params := make([]interface{}, 1+len(statuses))
	params[0] = squadID
	for i, status := range statuses {
		params[i+1] = models.NewDBFromInvitationStatus(status)
	}

	rows, err := d.getTxIfJoined(ctx).QueryContext(ctx, query, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get invitations")
	}
	defer d.closeRows(rows)

	var invitations []*models.DBInvitation
	for rows.Next() {
		invitation := models.DBInvitation{}
		err := rows.Scan(
			&invitation.ID,
			&invitation.SquadID,
			&invitation.SenderID,
			&invitation.RecipientID,
			&invitation.Status,
			&invitation.ReasonRejected,
			&invitation.CreatedAt,
			&invitation.UpdatedAt,
		)
		if err != nil {
			return nil, errors.Wrapf(err, "could not get invitation for squad ID %v and statuses %v", squadID, statuses)
		}
		invitations = append(invitations, &invitation)
	}

	return invitations, nil
}

// GetInvitationsBySquadID records stats for getInvitationsBySquadID.
func (d *datastore) GetInvitationsBySquadID(ctx context.Context, squadID string, statuses []models.InvitationStatus) ([]*models.DBInvitation, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "get_invitations_by_squad_id", startTime, err == nil)
	}(time.Now())

	invitations, err := d.getInvitationsBySquadID(ctx, squadID, statuses)
	return invitations, err
}
