package datastore

import (
	"context"
	"database/sql"
	"fmt"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// getInvitationsCountByRecipientID returns a count of invitations to the given channel with the given statuses in the db representation.
func (d *datastore) getInvitationsCountByRecipientID(ctx context.Context, channelID string, statuses []models.InvitationStatus) (int64, error) {
	if len(statuses) == 0 {
		return 0, errors.New("no statuses given")
	}

	query := fmt.Sprintf(`
		SELECT count(*)
		FROM squad_invitations
		WHERE recipient_id = $1
		AND status in (%s)`,
		generatePlaceholders(2, len(statuses)))

	params := make([]interface{}, 1+len(statuses))
	params[0] = channelID
	for i, status := range statuses {
		params[i+1] = models.NewDBFromInvitationStatus(status)
	}

	row := d.getTxIfJoined(ctx).QueryRowContext(ctx, query, params...)

	var count int64
	err := row.Scan(
		&count,
	)
	switch {
	case err == sql.ErrNoRows:
		return 0, nil
	case err != nil:
		return 0, errors.Wrapf(err, "could not get total count of invitations for channel ID %v and statuses %v", channelID, statuses)
	}

	return count, nil
}

// GetInvitationsCountByRecipientID records stats for getInvitationsCountByRecipientID.
func (d *datastore) GetInvitationsCountByRecipientID(ctx context.Context, channelID string, statuses []models.InvitationStatus) (int64, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "get_invitations_count_by_recipient_id", startTime, err == nil)
	}(time.Now())

	count, err := d.getInvitationsCountByRecipientID(ctx, channelID, statuses)
	return count, err
}
