package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// getInvitePolicyByChannelID returns db representation of the invite policy with the provided channel ID.
func (d *datastore) getInvitePolicyByChannelID(ctx context.Context, channelID string) (*models.DBInvitePolicy, error) {
	query := `
		SELECT channel_id, caller_id, invite_policy, created_at, updated_at
		FROM squad_invite_policy
		WHERE channel_id = $1`

	row := d.getTxIfJoined(ctx).QueryRowContext(ctx, query, channelID)
	invitePolicy := models.DBInvitePolicy{}
	err := row.Scan(
		&invitePolicy.ChannelID,
		&invitePolicy.CallerID,
		&invitePolicy.InvitePolicy,
		&invitePolicy.CreatedAt,
		&invitePolicy.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return &models.DBInvitePolicy{
			ChannelID:    channelID,
			CallerID:     channelID,
			InvitePolicy: string(models.InvitationPolicyNetwork),
		}, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not get invite policy for channel ID %v", channelID)
	}

	return &invitePolicy, nil
}

// GetInvitePolicyByChannelID records stats for getInvitePolicyByChannelID.
func (d *datastore) GetInvitePolicyByChannelID(ctx context.Context, channelID string) (*models.DBInvitePolicy, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "get_invite_policy_by_channel_id", startTime, err == nil)
	}(time.Now())

	invitePolicy, err := d.getInvitePolicyByChannelID(ctx, channelID)
	return invitePolicy, err
}
