package datastore

import (
	"context"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// getMembersBySquadID returns a list of members of the squad in the db representation.
func (d *datastore) getMembersBySquadID(ctx context.Context, squadID string) ([]*models.DBMember, error) {
	query := `
		SELECT id, squad_id, member_id, created_at, deleted_at, status
		FROM squad_members
		WHERE squad_id = $1 AND status <> 'deleted'
		ORDER BY created_at`

	rows, err := d.getTxIfJoined(ctx).QueryContext(ctx, query, squadID)
	if err != nil {
		return nil, errors.Wrap(err, "could not get events")
	}
	defer d.closeRows(rows)

	var members []*models.DBMember
	for rows.Next() {
		member := models.DBMember{}
		err := rows.Scan(
			&member.ID,
			&member.SquadID,
			&member.MemberID,
			&member.CreatedAt,
			&member.DeletedAt,
			&member.Status,
		)
		if err != nil {
			return nil, errors.Wrapf(err, "could not get members for squad ID %v", squadID)
		}
		members = append(members, &member)
	}

	return members, nil
}

// GetMembersBySquadID records stats for getMembersBySquadID.
func (d *datastore) GetMembersBySquadID(ctx context.Context, squadID string) ([]*models.DBMember, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "get_members_by_squad_id", startTime, err == nil)
	}(time.Now())

	members, err := d.getMembersBySquadID(ctx, squadID)
	return members, err
}
