package datastore

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// getMembersByStatuses returns a list of members with the given statuses in the db representation.
func (d *datastore) getMembersByStatuses(ctx context.Context, statuses []models.MemberStatus, squadStatuses []models.SquadStatus) ([]*models.DBMemberAndSquad, error) {
	if len(statuses) == 0 || len(squadStatuses) == 0 {
		return nil, errors.New("no statuses given")
	}

	query := fmt.Sprintf(`
		SELECT m.id, m.squad_id, m.member_id, m.created_at, m.deleted_at, m.status, s.id, s.owner_id, s.status, s.created_by, s.created_at, s.updated_at
		FROM squad_members m, squads s
		WHERE m.status in (%s)
		AND s.status in (%s)
		AND m.squad_id = s.id
		ORDER BY m.created_at`,
		generatePlaceholders(1, len(statuses)),
		generatePlaceholders(1+len(statuses), len(squadStatuses)))

	params := make([]interface{}, len(statuses)+len(squadStatuses))
	for i, status := range statuses {
		params[i] = models.NewDBFromMemberStatus(status)
	}
	for i, status := range squadStatuses {
		params[i+len(statuses)] = models.NewDBFromSquadStatus(status)
	}

	rows, err := d.getTxIfJoined(ctx).QueryContext(ctx, query, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get members")
	}
	defer d.closeRows(rows)

	var members []*models.DBMemberAndSquad
	for rows.Next() {
		member := models.DBMemberAndSquad{
			Member: &models.DBMember{},
			Squad:  &models.DBSquad{},
		}
		err := rows.Scan(
			&member.Member.ID,
			&member.Member.SquadID,
			&member.Member.MemberID,
			&member.Member.CreatedAt,
			&member.Member.DeletedAt,
			&member.Member.Status,
			&member.Squad.ID,
			&member.Squad.OwnerID,
			&member.Squad.Status,
			&member.Squad.CreatedBy,
			&member.Squad.CreatedAt,
			&member.Squad.UpdatedAt,
		)
		if err != nil {
			return nil, errors.Wrapf(err, "could not get members for statuses %v", statuses)
		}
		members = append(members, &member)
	}

	return members, nil
}

// GetMembersByStatuses records stats for getMembersByStatuses.
func (d *datastore) GetMembersByStatuses(ctx context.Context, statuses []models.MemberStatus, squadStatuses []models.SquadStatus) ([]*models.DBMemberAndSquad, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "get_members_by_statuses", startTime, err == nil)
	}(time.Now())

	members, err := d.getMembersByStatuses(ctx, statuses, squadStatuses)
	return members, err
}
