package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// getSquadByChannelID returns the db representation of the squad the provided channel belongs to.
func (d *datastore) getSquadByChannelID(ctx context.Context, channelID string) (*models.DBSquad, error) {
	query := `
		SELECT s.id, s.owner_id, s.status, s.created_by, s.created_at, s.updated_at 
		FROM squad_members m, squads s
		WHERE m.member_id = $1 AND m.deleted_at IS NULL AND m.squad_id = s.id AND s.status != 'ended'`

	row := d.getTxIfJoined(ctx).QueryRowContext(ctx, query, channelID)

	squad := models.DBSquad{}
	err := row.Scan(
		&squad.ID,
		&squad.OwnerID,
		&squad.Status,
		&squad.CreatedBy,
		&squad.CreatedAt,
		&squad.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not get squad for channel ID %v", channelID)
	}

	return &squad, nil
}

// GetSquadByChannelID records stats for getSquadByChannelID.
func (d *datastore) GetSquadByChannelID(ctx context.Context, channelID string) (*models.DBSquad, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "get_squad_by_channel_id", startTime, err == nil)
	}(time.Now())

	squad, err := d.getSquadByChannelID(ctx, channelID)
	return squad, err
}
