package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// getSquadByID returns the db representation of the squad with the provided ID.
func (d *datastore) getSquadByID(ctx context.Context, id string) (*models.DBSquad, error) {
	query := `
		SELECT id, owner_id, status, created_by, created_at, updated_at 
		FROM squads
		WHERE id = $1`

	row := d.getTxIfJoined(ctx).QueryRowContext(ctx, query, id)

	squad := models.DBSquad{}
	err := row.Scan(
		&squad.ID,
		&squad.OwnerID,
		&squad.Status,
		&squad.CreatedBy,
		&squad.CreatedAt,
		&squad.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not get squad for ID %v", id)
	}

	return &squad, nil
}

// GetSquadByID records stats for getSquadByID.
func (d *datastore) GetSquadByID(ctx context.Context, id string) (*models.DBSquad, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "get_squad_by_id", startTime, err == nil)
	}(time.Now())

	squad, err := d.getSquadByID(ctx, id)
	return squad, err
}
