package datastore

import (
	"context"
	"time"

	"code.justin.tv/feeds/errors"
)

// hardDeleteExpiredInvitations hard deletes all soft deleted invitations before the given time in the database.
func (d *datastore) hardDeleteExpiredInvitations(ctx context.Context, deleteBefore time.Time) (int64, error) {
	query := `
		DELETE FROM squad_invitations
		WHERE status IN ('accepted','deleted')
		AND updated_at <= $1`
	result, err := d.getTxIfJoined(ctx).ExecContext(ctx, query, ConvertToDBTime(deleteBefore))
	if err != nil {
		return 0, errors.Wrapf(err, "could not hard delete expired invitations before %v", deleteBefore)
	}

	rowsDeleted, err := result.RowsAffected()
	if err != nil {
		d.log.LogCtx(ctx, "err", err, "error getting squad invitations rows deleted")
	}
	return rowsDeleted, nil
}

// HardDeleteExpiredInvitations records stats for hardDeleteExpiredInvitations.
func (d *datastore) HardDeleteExpiredInvitations(ctx context.Context, deleteBefore time.Time) (int64, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "hard_delete_expired_invitations", startTime, err == nil)
	}(time.Now())

	rowsDeleted, err := d.hardDeleteExpiredInvitations(ctx, deleteBefore)
	return rowsDeleted, err
}
