package datastore

import (
	"context"
	"time"

	"code.justin.tv/feeds/errors"
)

// hardDeleteExpiredMembers hard deletes all soft deleted members before the given time in the database.
func (d *datastore) hardDeleteExpiredMembers(ctx context.Context, deleteBefore time.Time) (int64, error) {
	query := `
		DELETE FROM squad_members
		WHERE status = 'deleted'
		AND deleted_at <= $1`
	result, err := d.getTxIfJoined(ctx).ExecContext(ctx, query, ConvertToDBTime(deleteBefore))
	if err != nil {
		return 0, errors.Wrapf(err, "could not hard delete expired members before %v", deleteBefore)
	}

	rowsDeleted, err := result.RowsAffected()
	if err != nil {
		d.log.LogCtx(ctx, "err", err, "error getting squad members rows deleted")
	}
	return rowsDeleted, nil
}

// HardDeleteExpiredMembers records stats for hardDeleteExpiredMembers.
func (d *datastore) HardDeleteExpiredMembers(ctx context.Context, deleteBefore time.Time) (int64, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "hard_delete_expired_members", startTime, err == nil)
	}(time.Now())

	rowsDeleted, err := d.hardDeleteExpiredMembers(ctx, deleteBefore)
	return rowsDeleted, err
}
