package datastore

import (
	"context"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// hardDeleteExpiredSquads hard deletes all soft deleted squads before the given time in the database.
func (d *datastore) hardDeleteExpiredSquads(ctx context.Context, deleteBefore time.Time) ([]*models.DBSquad, error) {
	query := `
		DELETE FROM squads 
		WHERE status = 'ended'
		AND updated_at <= $1
		RETURNING id, owner_id, status, created_by, created_at, updated_at`
	rows, err := d.getTxIfJoined(ctx).QueryContext(ctx, query, ConvertToDBTime(deleteBefore))
	if err != nil {
		return nil, errors.Wrapf(err, "could not hard delete expired squads before %v", deleteBefore)
	}

	defer d.closeRows(rows)

	var squads []*models.DBSquad
	for rows.Next() {
		squad := models.DBSquad{}
		err := rows.Scan(
			&squad.ID,
			&squad.OwnerID,
			&squad.Status,
			&squad.CreatedBy,
			&squad.CreatedAt,
			&squad.UpdatedAt,
		)
		if err != nil {
			return nil, errors.Wrapf(err, "could not hard delete expired squads before %v", deleteBefore)
		}
		squads = append(squads, &squad)
	}

	return squads, nil
}

// HardDeleteExpiredSquads records stats for hardDeleteExpiredSquads.
func (d *datastore) HardDeleteExpiredSquads(ctx context.Context, deleteBefore time.Time) ([]*models.DBSquad, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "hard_delete_expired_squads", startTime, err == nil)
	}(time.Now())

	squads, err := d.hardDeleteExpiredSquads(ctx, deleteBefore)
	return squads, err
}
