package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
)

// hardDeleteInvitePolicyByChannelID deletes the channel's invite policy from the database.
func (d *datastore) hardDeleteInvitePolicyByChannelID(ctx context.Context, channelID string) error {
	query := `
		DELETE FROM squad_invite_policy
		WHERE channel_id = $1`

	params := make([]interface{}, 1)
	params[0] = channelID

	_, err := d.getTxIfJoined(ctx).ExecContext(ctx, query, params...)

	if err == sql.ErrNoRows {
		return nil
	} else if err != nil {
		return errors.Wrapf(err, "could not hard delete invite policy for ID %v", channelID)
	}

	return nil
}

// HardDeleteInvitePolicyByChannelID records stats for hardDeleteInvitePolicyByChannelID.
func (d *datastore) HardDeleteInvitePolicyByChannelID(ctx context.Context, channelID string) error {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "hard_delete_invite_policy_by_channel_id", startTime, err == nil)
	}(time.Now())

	err = d.hardDeleteInvitePolicyByChannelID(ctx, channelID)
	return err
}
